package com.xebialabs.xlrelease.reports.excel;

import java.awt.Color;
import java.io.IOException;
import java.net.URL;
import java.util.Date;
import java.util.Objects;
import org.apache.commons.io.IOUtils;
import org.apache.poi.common.usermodel.HyperlinkType;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.CellReference;
import org.apache.poi.ss.util.CellUtil;
import org.apache.poi.xssf.usermodel.*;
import org.joda.time.Duration;
import org.joda.time.PeriodType;
import org.joda.time.format.PeriodFormatter;
import org.joda.time.format.PeriodFormatterBuilder;

import com.xebialabs.xlrelease.domain.Phase;

import static com.xebialabs.xlrelease.reports.audit.CommonFormat.DATE_FORMAT;
import static org.apache.poi.util.Units.EMU_PER_PIXEL;

/**
 * This is an abstraction of the actual excel writer
 */
public class ExcelSheetWriter {

    private static final String DEFAULT_PHASE_COLOR = "#E7E6E6";
    public static final Integer MAXIMUM_CELL_LENGTH = 32_767;
    public static final PeriodFormatter PERIOD_FORMATTER = new PeriodFormatterBuilder()
            .printZeroAlways()
            .appendDays().appendSuffix("d")
            .appendSeparator(" ")
            .appendHours().appendSuffix("h")
            .appendSeparator(" ")
            .appendMinutes().appendSuffix("m")
            .appendSeparator(" ")
            .appendSeconds().appendSuffix("s")
            .toFormatter();

    private final ReportWorkbook reportWorkbook;
    private final XSSFWorkbook workbook;
    private final XSSFSheet sheet;
    private final ExcelStyles styles;
    private XSSFRow row;

    int rowIndex = 0;
    int columnIndex = 0;
    private int groupStartIndex;

    private XSSFCellStyle dateStyle;
    private XSSFCellStyle phaseStyle;
    private XSSFCellStyle cellStylePhase;
    private XSSFCellStyle cellStyleDefault;

    boolean isPhaseRow = false;
    boolean isBoldRow = false;

    boolean isWrappedRow = false;
    private float defaultRowHeight = 0;


    public ExcelSheetWriter(ReportWorkbook reportWorkbook, XSSFSheet sheet, ExcelStyles styles) {
        this.reportWorkbook = reportWorkbook;
        this.sheet = sheet;
        this.workbook = reportWorkbook.getWorkbookWithoutErrorSheet();
        this.styles = styles;
        dateStyle = workbook.createCellStyle();
        dateStyle.setDataFormat(workbook.createDataFormat().getFormat(DATE_FORMAT()));

        phaseStyle = workbook.createCellStyle();
        phaseStyle.setFillForegroundColor(new XSSFColor(Color.decode(DEFAULT_PHASE_COLOR)));
        phaseStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);

        sheet.protectSheet("");
    }

    public ExcelSheetWriter newRow() {
        row = sheet.createRow(rowIndex++);
        columnIndex = 0;

        return this;
    }

    public ExcelSheetWriter setRowHeight(float points) {
        row.setHeightInPoints(points);
        return this;
    }

    public ExcelSheetWriter indent() {
        columnIndex++;
        return this;
    }

    public ExcelSheetWriter goToStartOfLine() {
        columnIndex = 0;
        return this;
    }

    public ExcelSheetWriter newColumn() {
        CellUtil.createCell(row, columnIndex++, "", getDefaultStyle());
        return this;
    }

    public ExcelSheetWriter addHeaderCell(String text, int width) {
        Cell headerCell = CellUtil.createCell(row, columnIndex, truncate(text));
        headerCell.setCellStyle(getHeaderCellStyle());
        sheet.setColumnWidth(columnIndex, numberOfCharacter(width));
        columnIndex++;

        return this;
    }

    public ExcelSheetWriter addHeaderCell(ExcelHeaderColumn header) {
        return addHeaderCell(header.header(), header.width());
    }

    public ExcelSheetWriter addHeaderCell(ExcelHeaderColumn header, XSSFCellStyle style) {
        return addHeaderCell(header.header(), header.width(), style);
    }

    // TODO: abstract styles and make this private.
    public ExcelSheetWriter addHeaderCell(String text, int width, XSSFCellStyle style) {
        Cell headerCell = CellUtil.createCell(row, columnIndex, truncate(text));
        headerCell.setCellStyle(style);
        sheet.setColumnWidth(columnIndex, numberOfCharacter(width));
        columnIndex++;

        return this;
    }

    public ExcelSheetWriter addCell(Object entry) {
        if (entry instanceof Date) {
            addCell((Date) entry);
        } else if (entry instanceof Duration) {
            addCell((Duration) entry);
        } else {
            addCell(Objects.toString(entry, ""));
        }

        return this;
    }

    protected ExcelSheetWriter addCell(Object entry, XSSFCellStyle style) {
        if (entry instanceof Date) {
            addCell((Date) entry, style);
        } else if (entry instanceof Duration) {
            addCell((Duration) entry, style);
        } else {
            addCell(Objects.toString(entry, ""), style);
        }
        return this;
    }

    public ExcelSheetWriter addCell(Integer entry, XSSFCellStyle style) {
        Cell cell = CellUtil.createCell(row, columnIndex, null, style);
        cell.setCellValue((double) entry);
        cell.setCellStyle(style);
        columnIndex++;
        return this;
    }

    public ExcelSheetWriter addCell(String text, XSSFCellStyle style) {
        CellUtil.createCell(row, columnIndex, truncate(text), style);
        columnIndex++;

        return this;
    }

    public ExcelSheetWriter addCell(String text) {
        CellUtil.createCell(row, columnIndex, truncate(text), getDefaultStyle());
        columnIndex++;

        return this;
    }

    public ExcelSheetWriter setCellHyperlink(XSSFHyperlink hyperlink) {
        row.getCell(columnIndex - 1).setHyperlink(hyperlink);
        return this;
    }


    public ExcelSheetWriter addCell(XSSFRichTextString text, XSSFCellStyle style) {
        XSSFCell textCell = row.createCell(columnIndex);
        textCell.setCellValue(text);
        textCell.setCellStyle(style);
        columnIndex++;

        return this;
    }

    public ExcelSheetWriter addCell(Date date, XSSFCellStyle style) {
        XSSFCell dateCell = row.createCell(columnIndex);
        dateCell.setCellStyle(style);
        if (date != null) {
            dateCell.setCellValue(date);
        }
        columnIndex++;
        return this;
    }

    public ExcelSheetWriter addCell(Date date) {
        addCell(date, getDateStyle());
        return this;
    }

    public ExcelSheetWriter addCell(Duration duration) {
        XSSFCell durationCell = row.createCell(columnIndex);
        durationCell.setCellStyle(getDefaultStyle());
        if (duration != null) {
            durationCell.setCellValue(truncate(PERIOD_FORMATTER.print(duration.toPeriod().normalizedStandard(PeriodType.dayTime()))));
        }
        columnIndex++;

        return this;
    }

    public ExcelSheetWriter addCell(String text, Color color) {

        XSSFColor backgroundColor = null;
        if (color != null) {
            backgroundColor = new XSSFColor(color);
        }

        CellUtil.createCell(row, columnIndex, truncate(text), getStyle(false, false, backgroundColor));

        columnIndex++;

        return this;
    }

    public ExcelSheetWriter addEmptyCell(XSSFCellStyle style) {
        if (row.getCell(columnIndex) == null) {
            XSSFCell cell = row.createCell(columnIndex);
            cell.setCellStyle(style);
        }
        columnIndex++;
        return this;
    }

    public ExcelSheetWriter addHyperlink(String text, String url, XSSFCellStyle hyperlinkStyle) {
        return addHyperlink(text, url, HyperlinkType.URL, hyperlinkStyle);
    }

    public ExcelSheetWriter addHyperlink(String text, String url, HyperlinkType type, XSSFCellStyle hyperlinkStyle) {
        Cell cell = CellUtil.createCell(row, columnIndex++, truncate(text), hyperlinkStyle);
        Hyperlink link = workbook.getCreationHelper().createHyperlink(type);
        link.setAddress(url);
        cell.setHyperlink(link);

        return this;
    }

    public ExcelSheetWriter addFullText(String text) {
        isWrappedRow = true;
        defaultRowHeight = CellUtil.getRow(row.getRowNum(), sheet).getHeightInPoints();

        CellUtil.createCell(row, columnIndex++, truncate(text), getDefaultStyle());
        CellUtil.getRow(row.getRowNum(), sheet).setHeightInPoints(defaultRowHeight);
        isWrappedRow = false;

        return this;
    }

    protected ExcelSheetWriter addProperty(String key, String value) {
        addCell(truncate(key));
        addCell(truncate(value));
        newRow();

        return this;
    }

    public ExcelSheetWriter mergeCells(CellRangeAddress range) {
        sheet.addMergedRegion(range);
        return this;
    }

    public ExcelSheetWriter applyPhaseColor(Phase phase) {
        Color color;
        String phaseColor = phase.getColor();
        phaseStyle = workbook.createCellStyle();
        if (phaseColor == null) {
            color = Color.decode(DEFAULT_PHASE_COLOR);
        } else {
            color = Color.decode(phaseColor).brighter(); // for readability purposes
        }
        phaseStyle.setFillForegroundColor(new XSSFColor(color));
        phaseStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        return this;
    }

    protected ExcelSheetWriter startGroup() {
        groupStartIndex = rowIndex;

        return this;
    }

    protected ExcelSheetWriter endGroup() {
        sheet.groupRow(groupStartIndex, rowIndex - 1);

        return this;
    }

    private ExcelSheetWriter addHeaderFilter(int headerRowIndex, int startIndex, int endIndex, boolean freezePane) {
        sheet.setAutoFilter(new CellRangeAddress(headerRowIndex, headerRowIndex, startIndex, endIndex));
        if (freezePane) {
            sheet.createFreezePane(0, 1);
        }

        return this;
    }

    public ExcelSheetWriter addHeaderFilter(int numberOfColumns, boolean freezePane) {
        return addHeaderFilter(rowIndex - 1, 0, numberOfColumns - 1, freezePane);
    }

    public ExcelSheetWriter addHeaderFilter(int numberOfColumns) {
        return addHeaderFilter(numberOfColumns, false);
    }

    //
    // Styles
    //

    public XSSFCellStyle getDefaultStyle() {
        if (isBoldRow) {
            return styles.bold();
        } else if (isPhaseRow) {
            return styles.addBackgroundColor(styles.defaultStyle(), Color.decode(DEFAULT_PHASE_COLOR));
        } else {
            return styles.defaultStyle();
        }
    }

    private XSSFCellStyle getHeaderCellStyle() {
        return styles.whiteOnGreen();
    }

    public XSSFCellStyle getDateStyle() {
        return getStyle(false, true, null);
    }

    private XSSFCellStyle getStyle(boolean isHeader, boolean isDate, XSSFColor backgroundColor) {
        XSSFCellStyle cellStyle = getStyleObject();
        cellStyle.setAlignment(HorizontalAlignment.LEFT);
        cellStyle.setVerticalAlignment(VerticalAlignment.TOP);
        XSSFFont defaultFont = workbook.createFont();
        defaultFont.setFontHeightInPoints((short) 12);
        cellStyle.setFont(defaultFont);

        if (isDate) {
            cellStyle.setDataFormat(dateStyle.getDataFormat());
        }

        if (backgroundColor != null) {
            cellStyle.setFillForegroundColor(backgroundColor);
            cellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        } else if (isPhaseRow) {
            cellStyle.setFillForegroundColor(phaseStyle.getFillForegroundXSSFColor());
            cellStyle.setFillPattern(phaseStyle.getFillPatternEnum());
        }

        if (isWrappedRow) {
            cellStyle.setWrapText(true);
        }

        if (isBoldRow) {
            XSSFFont font = workbook.createFont();
            font.setBold(true);
            cellStyle.setFont(font);
        }

        if (isHeader) {
            cellStyle.setAlignment(HorizontalAlignment.CENTER);
        }

        return cellStyle;
    }

    private XSSFCellStyle getStyleObject() {
        if(isPhaseRow) {
            if (cellStylePhase == null) {
                cellStylePhase = workbook.createCellStyle();
            }
            return cellStylePhase;
        } else {
            if (cellStyleDefault == null) {
                cellStyleDefault = workbook.createCellStyle();
            }
            return cellStyleDefault;
        }
    }

    public ExcelSheetWriter insertImage(URL imageResource, int rowIndex, int colIndex, int magicWidth, int magicHeight) {
        byte[] imageBytes = null;

        try {
            imageBytes = IOUtils.toByteArray(imageResource);
        } catch (IOException e) {
            // log?
        }

        if (imageBytes != null) {
            XSSFClientAnchor anchor = workbook.getCreationHelper().createClientAnchor();
            anchor.setAnchorType(ClientAnchor.AnchorType.MOVE_AND_RESIZE);
            anchor.setCol1(colIndex);
            anchor.setRow1(rowIndex);
            anchor.setRow2(colIndex);
            anchor.setCol2(rowIndex);

            anchor.setDx1(30 * EMU_PER_PIXEL);
            anchor.setDx2(magicWidth * EMU_PER_PIXEL);
            anchor.setDy1(10 * EMU_PER_PIXEL);
            anchor.setDy2(magicHeight * EMU_PER_PIXEL);

            int pictureIndex = workbook.addPicture(imageBytes, Workbook.PICTURE_TYPE_PNG);

            XSSFPicture picture = sheet.createDrawingPatriarch().createPicture(anchor, pictureIndex);
            picture.resize(1, 0.9);

        }

        return this;
    }

    public ExcelSheetWriter replaceWithHyperlink(int srcRow, int srcCol, String targetSheet, int targetRow, int targetCol, XSSFCellStyle hyperlinkStyle) {
        XSSFCell cell = sheet.getRow(srcRow).getCell(srcCol);

        Hyperlink link = workbook.getCreationHelper().createHyperlink(HyperlinkType.DOCUMENT);

        link.setAddress(String.format("'%s'!%s%d", targetSheet, CellReference.convertNumToColString(targetCol), targetRow + 1));
        cell.setHyperlink(link);
        cell.setCellStyle(hyperlinkStyle);
        return this;
    }

    public ExcelSheetWriter addBottomBorderToCell(XSSFCell cell) {
        XSSFCellStyle style = workbook.createCellStyle();
        style.cloneStyleFrom(cell.getCellStyle());
        style.setBorderBottom(BorderStyle.THIN);
        cell.setCellStyle(style);
        return this;
    }

    /**
     * See XSSFSheet.setColumnWidth to understand the magic number 256
     */
    private static int numberOfCharacter(int number) {
        return number * 256;
    }

    public XSSFWorkbook getWorkbook() {
        Export.preventDynamicDataExchangeAttack(this.workbook);
        return this.workbook;
    }

    public XSSFSheet getSheet() {
        return this.sheet;
    }

    public int getRowIndex() {
        return rowIndex;
    }

    public int getColumnIndex() {
        return columnIndex;
    }

    public boolean isWrappedRow() {
        return isWrappedRow;
    }

    public void setWrappedRow(boolean wrapped) {
        this.isWrappedRow = wrapped;
    }

    public boolean isPhaseRow() {
        return isPhaseRow;
    }

    public void setPhaseRow(boolean isPhaseRow) {
        this.isPhaseRow = isPhaseRow;
    }

    public ReportWorkbook getReportWorkbook() {
        return reportWorkbook;
    }

    public static String truncate(String text) {
        if (text != null && text.length() > MAXIMUM_CELL_LENGTH) {
            return text.substring(0, MAXIMUM_CELL_LENGTH);
        } else {
            return text;
        }
    }

}
