package com.xebialabs.xlrelease.reports.repository.codecompliance

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters
import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.udm.reporting.CodeComplianceRecord
import com.xebialabs.xlrelease.domain.utils.FullReleaseId
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.reports.repository.CommonRowAndParamsMapper.{commonFieldNames, commonParameterNames}
import com.xebialabs.xlrelease.reports.repository.codecompliance.CodeComplianceRecordArchiveRepository._
import com.xebialabs.xlrelease.reports.repository.{ArchivedTasksRepository, RepositoryExceptionUtils, SearchRecordsSupport}
import com.xebialabs.xlrelease.repository.FacetRepository
import com.xebialabs.xlrelease.repository.FacetRepository.SpecializedFacetRepository
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.{CiId, _}
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.collection.mutable

object CodeComplianceRecordArchiveRepository {

  import CODE_COMPLIANCE_TASK_REPORTING_RECORD_ARCHIVE._

  private lazy val alias = "c"
  private lazy val ALL_FIELDS =
    s""" ${commonFieldNames(CODE_COMPLIANCE_TASK_REPORTING_RECORD_ARCHIVE, alias)},
       | t.$REPORT_TASKS_ID_COLUMN,
       | $alias.$PROJECT,
       | $alias.$PROJECT_URL,
       | $alias.$ANALYSIS_DATE,
       | $alias.$OUTCOME,
       | $alias.$COMPLIANCE_DATA""".stripMargin

  private lazy val SELECT_ALL_CODE_COMPLIANCE_RECORDS =
    s"""SELECT
       | $ALL_FIELDS
       |FROM $TABLE $alias
       |JOIN $REPORT_TASKS_TABLE_NAME t ON t.$REPORT_TASKS_UID_COLUMN = $alias.$TASK_UID""".stripMargin

  private lazy val SELECT_CODE_COMPLIANCE_RECORD_BY_ID: String =
    s"""|$SELECT_ALL_CODE_COMPLIANCE_RECORDS
        |WHERE $alias.$RECORD_ID = :recordId""".stripMargin

  private lazy val SELECT_CODE_COMPLIANCE_RECORDS_BY_TASK_ID =
    s"""|$SELECT_ALL_CODE_COMPLIANCE_RECORDS
        |WHERE t.$REPORT_TASKS_ID_COLUMN = :taskId""".stripMargin

  private lazy val SELECT_CODE_COMPLIANCE_RECORDS_BY_RELEASE_ID: String =
    s"""|$SELECT_ALL_CODE_COMPLIANCE_RECORDS
        |JOIN $REPORT_RELEASES_TABLE_NAME r ON r.$REPORT_RELEASES_ID_COLUMN = t.$REPORT_TASKS_RELEASEID_COLUMN
        |WHERE r.$REPORT_RELEASES_ID_COLUMN = :releaseId AND r.$REPORT_RELEASES_PRE_ARCHIVED <> 1""".stripMargin

  private lazy val EXISTS_CODE_COMPLIANCE_RECORD_BY_ID: String =
    s"""SELECT COUNT($RECORD_ID) FROM $TABLE WHERE $RECORD_ID = :recordId""".stripMargin

  private lazy val INSERT_CODE_COMPLIANCE_RECORD =
    s"""INSERT INTO $TABLE (
       | ${commonFieldNames(CODE_COMPLIANCE_TASK_REPORTING_RECORD_ARCHIVE)},
       | $TASK_UID,
       | $PROJECT,
       | $PROJECT_URL,
       | $ANALYSIS_DATE,
       | $OUTCOME,
       | $COMPLIANCE_DATA
       |) VALUES (
       | ${commonParameterNames()},
       | :taskUid,
       | :project,
       | :projectUrl,
       | :analysisDate,
       | :outcome,
       | :complianceData
       |)""".stripMargin

  private lazy val DELETE_CODE_COMPLIANCE_RECORD: String =
    s"""|DELETE FROM $TABLE
        |WHERE $RECORD_ID = :recordId""".stripMargin
}

class CodeComplianceRecordArchiveRepository(archivedTasksRepository: ArchivedTasksRepository,
                                            val ciIdService: CiIdService,
                                            val jdbcTemplate: JdbcTemplate,
                                            val dialect: Dialect) extends SpecializedFacetRepository(Type.valueOf(classOf[CodeComplianceRecord]))
  with FacetRepository.ForArchive
  with PersistenceSupport
  with SearchRecordsSupport[CodeComplianceRecord]
  with RepositoryExceptionUtils
  with Logging {

  override def get(recordId: String, recordType: Option[Type]): Facet = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(SELECT_CODE_COMPLIANCE_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), codeComplianceRowMapper)
      .headOption.getOrElse(throw new NotFoundException(s"Facet '$recordId' not found."))
  }

  override protected def doCreate(facet: Facet): Facet = {
    logger.debug("Creating CodeCompliance record in archive")
    requiresInstanceOfSupportedType(facet.getType)
    archivedTasksRepository.taskUidById(shortenId(facet.getTargetId)).map { taskUid =>
      val codeComplianceRecord = facet.asInstanceOf[CodeComplianceRecord]
      namedTemplate.update(INSERT_CODE_COMPLIANCE_RECORD, asQueryParameters(codeComplianceRecord, taskUid))
      facet
    }.getOrElse {
      throw targetTaskNotFoundException(facet.getId, facet.getTargetId)
    }
  }

  override def delete(recordId: String, recordType: Option[Type]): Unit =
    sqlUpdate(DELETE_CODE_COMPLIANCE_RECORD, params("recordId" -> getName(recordId.normalized)), _ => ())

  override def update(facet: Facet): Facet = throw new UnsupportedOperationException("You cannot update records!")

  override def search(facetsFilters: FacetFilters): Seq[Facet] = searchRecords(facetsFilters, SELECT_ALL_CODE_COMPLIANCE_RECORDS, codeComplianceRowMapper)

  override def exists(recordId: String, recordType: Option[Type]): Boolean = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(EXISTS_CODE_COMPLIANCE_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), _.getInt(1) > 0).headOption.exists(identity)
  }

  override def findAllFacetsByRelease(release: Release): Seq[Facet] = findAllFacetsByReleaseId(release.getId).toSeq

  override def findAllFacetsByTask(task: Task): Seq[Facet] = findAllFacetsByTargetId(shortenId(task.getId)).toSeq


  override protected def findAllFacetsByTargetId(targetId: CiId): mutable.Seq[CodeComplianceRecord] =
    sqlQuery(SELECT_CODE_COMPLIANCE_RECORDS_BY_TASK_ID, params("taskId" -> targetId), codeComplianceRowMapper)

  override protected def findAllFacetsByReleaseId(releaseId: CiId): mutable.Seq[CodeComplianceRecord] =
    sqlQuery[CodeComplianceRecord](SELECT_CODE_COMPLIANCE_RECORDS_BY_RELEASE_ID, params(
      "releaseId" -> FullReleaseId(releaseId).withOnlyOneParentOrApplicationsForArchiveDb()
    ), codeComplianceRowMapper)

  private val codeComplianceRowMapper: RowMapper[CodeComplianceRecord] = (rs: ResultSet, _: Int) => {
    val codeComplianceRecord: CodeComplianceRecord = Type.valueOf(rs.getString(CODE_COMPLIANCE_TASK_REPORTING_RECORD_ARCHIVE.RECORD_TYPE))
      .getDescriptor.newInstance[CodeComplianceRecord](rs.getString(CODE_COMPLIANCE_TASK_REPORTING_RECORD_ARCHIVE.RECORD_ID))
    codeComplianceItemRowMapper(rs, codeComplianceRecord, rs.getString(REPORT_TASKS_ID_COLUMN), CODE_COMPLIANCE_TASK_REPORTING_RECORD_ARCHIVE)
  }
}
