package com.xebialabs.xlrelease.reports.audit

import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.xlplatform.io.ZipUtils
import com.xebialabs.xlrelease.api.v1.forms.ReleaseOrderMode
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.utils.{AdaptiveReleaseId, ReleaseIdInDatabase}
import com.xebialabs.xlrelease.reports.audit
import com.xebialabs.xlrelease.reports.audit.MultiReleaseAuditReportComponent._
import com.xebialabs.xlrelease.reports.domain.MaybeData
import com.xebialabs.xlrelease.reports.domain.MaybeData._
import com.xebialabs.xlrelease.reports.excel.MultiReleaseAuditReport.PerReleaseData
import com.xebialabs.xlrelease.reports.excel.{AuditReport, ExcelSheetWriter, MultiReleaseAuditReport}
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.reports.job.api.ReportJobRunContext
import com.xebialabs.xlrelease.reports.job.api.ReportStorage.StreamingReportStorage
import com.xebialabs.xlrelease.reports.service.ReportsService
import com.xebialabs.xlrelease.service.{ReleaseSearchService, ReleaseService, TaskAccessService}
import com.xebialabs.xlrelease.udm.reporting.AuditReportRequest
import com.xebialabs.xlrelease.views.converters.ReleaseViewConverter
import grizzled.slf4j.Logging
import org.apache.commons.io.FileUtils
import org.apache.poi.ss.usermodel.Workbook
import org.joda.time.PeriodType
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.io.File
import java.nio.file.{Path, Paths}
import java.text.SimpleDateFormat
import java.util.regex.Pattern
import java.util.{Collections, Date}
import scala.jdk.CollectionConverters._
import scala.util.Success

object MultiReleaseAuditReportComponent {

  val EXCEL_EXTENSION = ".xlsx"
  val EXCEL_CONTENT_TYPE = "application/vnd.ms-excel"
  val ZIP_EXTENSION = ".zip"
  val MASTER_TITLE = "Digital_ai Release Audit Report"
  val MASTER_TITLE_SEPARATOR = " - "
  val SINGLE_TITLE_SEPARATOR = " - "
  val SINGLE_REPORTS_DIR = "Reports"
  val DATETIME_FORMATTER = new SimpleDateFormat("yyyy-MM-dd_HHmmss")
  val FILENAME_MAX_LENGTH = 100

  val DEFAULT_PAGE_SIZE = 15
  val DEFAULT_ORDER_MODE = ReleaseOrderMode.start_date

  def sanitizeFilename(filename: String): String = { //Shouldn't be here!
    val p: Pattern = Pattern.compile("""[^\w \)\(]""", Pattern.UNICODE_CHARACTER_CLASS) // NOT Using """[\\%\/\?*:|\"<>]"""
    p.matcher(filename.slice(0, FILENAME_MAX_LENGTH)).replaceAll("_")
  }

  case class Data(generatedBy: String,
                  generatedOn: Date,
                  reportFilters: Seq[ReportFilter])

}

@Service
class MultiReleaseAuditReportComponent @Autowired()(reportsService: ReportsService,
                                                    releaseSearchService: ReleaseSearchService,
                                                    releaseService: ReleaseService,
                                                    serverConfiguration: ServerConfiguration,
                                                    releaseViewConverter: ReleaseViewConverter,
                                                    taskAccessService: TaskAccessService,
                                                    reportFilterMapper: ReportFilterMapper) extends Logging {

  def generateReport(reportDefinition: audit.MultiReleaseAuditReport, reportJobRunContext: ReportJobRunContext): File = {
    val data: MultiReleaseAuditReportComponent.Data = MultiReleaseAuditReportComponent.Data(
      reportDefinition.getGeneratedBy(),
      reportDefinition.getGeneratedOn(),
      reportDefinition.filters.asScala.toSeq
    )
    val releaseIds: Seq[ReleaseIdInDatabase] = releaseSearchService.searchAllIdsWithoutPermissionsCheck(data.reportFilters, Some(ReleaseOrderMode.start_date))
    logger.debug(s"Creating report for ${releaseIds.length} releases in job with id '${reportJobRunContext.jobId}'.")
    val totalWorkItems = releaseIds.length + 3

    val progressMonitor = reportJobRunContext.reportJobProgressMonitor
    progressMonitor.sendTotalWorkItems(totalWorkItems)

    val workDir = reportJobRunContext.reportStorage.localWorkDir

    val reportName = reportJobRunContext.reportName
    val reportDirName = reportName

    val reportDir: Path = Paths.get(workDir.getPath, reportDirName)
    val reportsDir: Path = reportDir.resolve(SINGLE_REPORTS_DIR)
    reportsDir.toFile.mkdirs()

    val releasesData: Map[AdaptiveReleaseId, Maybe[PerReleaseData]] = {
      releaseIds.foldLeft(Map.empty[AdaptiveReleaseId, Maybe[PerReleaseData]]) {
        case (done, id) =>
          val toR = done + (id.adaptiveReleaseId -> MaybeData.apply(generateSingleReleaseReport(reportJobRunContext.reportStorage, reportsDir, id)))
          progressMonitor.sendCompletedWorkItems(toR.keys.size)
          toR
      }
    }

    logger.debug(s"Creating master report with all releases")
    val masterData = MultiReleaseAuditReport.Data(
      data.generatedBy,
      data.generatedOn,
      instanceData,
      reportFilterMapper.map(data.reportFilters),
      releasesData.values.toSeq
    )
    val masterWorkBook: Workbook = MultiReleaseAuditReport.getWorkBook(masterData)
    val masterFile = reportJobRunContext.reportStorage.writeToLocalFile(reportDir, masterWorkBook, data.generatedOn, MASTER_TITLE)
    progressMonitor.sendCompletedWorkItems(releaseIds.length + 1)

    val fullReport: File = ZipUtils.archive(
      masterFile +: releasesData.values.collect {
        case Success(releaseData) =>
          releaseData.getValue.storedPath.get._1
      }.toSeq,
      reportDir.resolve(s"$reportName$ZIP_EXTENSION").toFile,
      stripPrefix = Some(workDir.getPath)
    )
    progressMonitor.sendCompletedWorkItems(releaseIds.length + 2)

    //You might wanna save some of the single audit reports at this moment
    FileUtils.deleteDirectory(reportsDir.toFile)
    masterFile.delete()

    progressMonitor.sendCompletedWorkItems(totalWorkItems)

    fullReport
  }

  private def generateSingleReleaseReport(reportStorage: StreamingReportStorage, workDir: Path, releaseId: ReleaseIdInDatabase): PerReleaseData = {
    try {
      logger.debug(s"Working on report for [${releaseId.adaptiveReleaseId.folderlessReleaseId()}]")
      val release: Release = reportsService.findReleaseByReleaseIdInDatabase(releaseIdInDatabase = releaseId)
      val data: AuditReport.Data = reportsService.fetchAuditReportData(release)
      val workBook = AuditReport.getWorkBook(data)
      val folder = reportsService.formattedReleaseFolderPath(release.getId)

      val reportFileName = if (data.releaseTree.release.isWorkflow) {
        s"${data.releaseTree.release.getTitle}_workflow"
      } else {
        data.releaseTree.release.getTitle
      }
      val stored: File = reportStorage.writeToLocalFile(workDir, workBook, data.releaseTree.release.getStartOrScheduledDate, reportFileName)

      val root = data.releaseTree.root
      PerReleaseData(
        title = release.getTitle,
        owner = release.getOwner,
        folder = folder,
        startDate = Option(release.getStartDate).getOrElse(release.getScheduledStartDate),
        endOrDueDate = root.endDate.map(dt => Left(dt.toDate)).orElse(root.dueDate.map(dt => Right(dt.toDate))).get, // potentially throws None.get exception
        duration = formatDuration(release),
        kind = release.getKind,
        status = release.getStatus,
        startedBy = data.releaseOverviewData.startedBy,
        storedPath = Some(stored, stored.getPath.stripPrefix(workDir.getParent.toFile.getPath).stripPrefix(File.separator)),
        templateInfo = data.releaseOverviewData.templateInfo
      )
    } catch {
      case e: Exception =>
        throw new IllegalStateException(s"Unable to generate individual report for ${releaseId.adaptiveReleaseId.folderlessReleaseId()}: ${e.getMessage}", e)
    }
  }

  private def instanceData: InstanceData = {
    val instanceName = reportsService.getInstanceName()
    val instanceUrl = Option(serverConfiguration.getServerUrl)
    val xlrVersion = reportsService.getInstanceVersion()
    InstanceData(instanceName, instanceUrl, xlrVersion)
  }

  def preview(reportDefinition: audit.MultiReleaseAuditReport): Any = {
    val reportRequest = AuditReportRequest(reportDefinition.filters.asScala.toSeq, Option(DEFAULT_ORDER_MODE))
    val releases = reportsService.getReleasesForPreview(reportRequest, 0, DEFAULT_PAGE_SIZE)
    val releaseCount = reportsService.getReleasesCountByStatus(reportRequest)
    val properties = Collections.emptyList[String]
    val extensions = Collections.singletonList("progress")
    val views = releases.asScala.map(r =>
      releaseViewConverter.toFullView(r,
        taskAccessService.getAllowedTaskTypesForAuthenticatedUser,
        properties,
        extensions,
        2
      )
    )
    Map("releases" -> views.asJava, "releaseCount" -> releaseCount).asJava
  }

  private def formatDuration(release: Release): String = {
    val duration = Option(release.getActualDuration).getOrElse(release.getComputedPlannedDuration)
    ExcelSheetWriter.PERIOD_FORMATTER.print(duration.toPeriod().normalizedStandard(PeriodType.dayTime()))
  }

}
