package com.xebialabs.xlrelease.reports.audit

import com.xebialabs.deployit.booter.local.utils.Strings
import com.xebialabs.deployit.security.{Role, RoleService}
import com.xebialabs.xlrelease.domain.Team
import com.xebialabs.xlrelease.domain.folder.Folder.ROOT_FOLDER_ID
import com.xebialabs.xlrelease.reports.domain.PermissionLabels
import com.xebialabs.xlrelease.reports.excel.PermissionReport
import com.xebialabs.xlrelease.reports.filters.CompositeFilter.Operator.AND
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.reports.job.api.{ReportJobProgressMonitor, ReportJobRunContext}
import com.xebialabs.xlrelease.reports.repository.{FolderUserPermissionsReportData, GlobalUserPermissionsReportData, UserPermissionsReportsRepository}
import com.xebialabs.xlrelease.reports.service.ReportsService
import com.xebialabs.xlrelease.repository.sql.persistence.FolderPersistence
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderRow
import com.xebialabs.xlrelease.service.{TeamService, UserProfileService}
import com.xebialabs.xlrelease.udm.reporting.filters.CompositeFilter
import com.xebialabs.xlrelease.utils.FolderId
import grizzled.slf4j.Logging
import org.apache.poi.ss.usermodel.Workbook
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.io.File
import java.nio.file.{Path, Paths}
import java.util
import scala.beans.BeanProperty
import scala.collection.mutable
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

@Service
class UserPermissionsReportComponent @Autowired()(reportsService: ReportsService,
                                                   userPermissionsReportsRepository: UserPermissionsReportsRepository,
                                                   folderPersistence: FolderPersistence,
                                                   teamService: TeamService,
                                                   roleService: RoleService,
                                                   reportFilterMapper: ReportFilterMapper,
                                                   userProfileService: UserProfileService) extends Logging {

  def generateReport(reportDefinition: UserPermissionsReport, reportJobRunContext: ReportJobRunContext): File = {

    val workDir = reportJobRunContext.reportStorage.localWorkDir
    val reportDirName = reportJobRunContext.reportName
    val reportDir: Path = Paths.get(workDir.getPath, reportDirName)
    reportDir.toFile.mkdirs()

    val workbook: Workbook = generateReportWorkbook(reportDefinition, reportJobRunContext.reportJobProgressMonitor)
    val reportFile = reportJobRunContext.reportStorage.writeToLocalFile(reportDir, workbook, reportDefinition.getGeneratedOn(), PermissionReport.REPORT_TITLE)

    reportFile
  }

  def generateReportWorkbook(reportDefinition: UserPermissionsReport, progressMonitor: ReportJobProgressMonitor): Workbook = {

    val filters: Option[Seq[ReportFilter]] = Option(reportDefinition.getFilters)
      .map((_: util.List[ReportFilter]).asScala.toSeq)

    val globalPermissionData: Iterable[GlobalUserPermissionsReportData] = getGlobalPermissionData()

    val folderPermissionData: Iterable[FolderUserPermissionsReportData] = getFolderPermissionData(getRolesByName, filters, Some(progressMonitor))

    val username = reportDefinition.getGeneratedBy()
    val userProfile = userProfileService.findByUsername(username)
    val data = PermissionReport.Data(
      username = if (Strings.isNotBlank(userProfile.getFullName)) userProfile.getFullName else username,
      creationDate = reportDefinition.getGeneratedOn(),
      instanceData = InstanceData(
        reportsService.getInstanceName(),
        reportsService.getServerUrl(),
        reportsService.getInstanceVersion()),
      globalPermissionData,
      folderPermissionData
    )

    val workbook: Workbook = PermissionReport.getWorkBook(data, filters.map(reportFilterMapper.map))

    progressMonitor.sendCompletedWorkItems(1)

    workbook
  }

  private def getRolesByName: Map[String, Role] = {
    roleService.readRoleAssignments().asScala.map((role: Role) => role.getName -> role).toMap
  }

  def getGlobalPermissionData(limit: Option[Long] = None): Iterable[GlobalUserPermissionsReportData] = {
    userPermissionsReportsRepository.generateGlobalReportData(limit)
  }

  def getFolderPermissionData(roles: Map[String, Role], filters: Option[Seq[ReportFilter]], progressMonitor: Option[ReportJobProgressMonitor]): Iterable[FolderUserPermissionsReportData] = {
    val filterVisitor = new UserPermissionsReportFilterVisitor()
    val compositeFilter: Option[CompositeFilter] = filters.map(f => new CompositeFilter(AND, f: _*))
    val folderRows: mutable.Buffer[FolderRow] = folderPersistence.findDescendantsById(ROOT_FOLDER_ID)
      .filterNot(_.folderId.absolute == ROOT_FOLDER_ID)
    val folderNamesById = folderRows.map(r => r.folderId.id -> r.name).toMap
    val folderFullNamesByIdCache = mutable.Map.empty[FolderId, String]

    def getFolderName(folderId: FolderId): String = {
      if (!folderFullNamesByIdCache.contains(folderId)) {
        folderFullNamesByIdCache.put(
          folderId,
          if (folderId.parent.absolute == ROOT_FOLDER_ID) {
            folderNamesById(folderId.id)
          } else {
            s"${getFolderName(folderId.parent)} / ${folderNamesById(folderId.id)}"
          })
      }
      folderFullNamesByIdCache(folderId)
    }

    // 1 for each folder, 1 for global and 1 for the XLSX file itself
    progressMonitor.foreach(_.sendTotalWorkItems(folderRows.size + 2))

    folderRows
      .filter((folder: FolderRow) => {
        compositeFilter match {
          case Some(f) =>
            filterVisitor.setFolderlessFolderId(folder.folderId.id)
            f.accept(filterVisitor)
          case _ => true
        }
      })
      .zipWithIndex
      .view
      .flatMap {
        case (folder: FolderRow, folderIndex: Int) =>
          val teams: mutable.Seq[Team] = teamService.getEffectiveTeams(folder.folderId.absolute).asScala

          val byUserAndPermission: mutable.Map[(String, String), ListBuffer[String]] =
            new mutable.HashMap[(String, String), ListBuffer[String]]()

          teams.foreach((team: Team) => {
            val rolesByPrincipal: mutable.Map[String, ListBuffer[String]] = new mutable.HashMap[String, ListBuffer[String]]
            val members: Set[String] = team.getMembers.asScala.toSet
            team.getRoles.asScala.foreach((role: String) => {
              if (roles.contains(role)) {
                roles(role).getPrincipals.asScala.foreach(principal => {
                  rolesByPrincipal.getOrElseUpdate(principal, new mutable.ListBuffer[String]) += role
                })
              }
            })
            val allPrincipals: Set[String] = members ++ rolesByPrincipal.keys
            team.getPermissions.asScala.foreach((permission: String) => {
              allPrincipals.foreach((principal: String) => {
                {
                  val isInTeam = if (members.contains(principal)) {
                    Seq(team.getTeamName)
                  } else {
                    Seq()
                  }
                  val roles = rolesByPrincipal.getOrElse(principal, Nil).map { roleName =>
                    s"${team.getTeamName} ($roleName)"
                  }
                  isInTeam ++ roles
                }.foreach(item => {
                  byUserAndPermission.getOrElseUpdate(principal -> permission, new ListBuffer[String]) += item
                })
              })
            })
          })

          progressMonitor.foreach(_.sendCompletedWorkItems(folderIndex + 1))

          byUserAndPermission.map(entity => {
            FolderUserPermissionsReportData(
              getFolderName(folder.folderId),
              entity._1._1,
              Option(userProfileService.findByUsername(entity._1._1)).map(_.getFullName).getOrElse(""),
              PermissionLabels.getLabel(entity._1._2),
              entity._2.toList
            )
          })
      }
  }

  def preview(reportDefinition: UserPermissionsReport): Any = {
    case class GlobalUserPermissionsViewRow(@BeanProperty principal: String,
                                            @BeanProperty userFullName: String,
                                            @BeanProperty permission: String,
                                            @BeanProperty roles: String)

    case class FolderUserPermissionsViewRow(@BeanProperty principal: String,
                                            @BeanProperty userFullName: String,
                                            @BeanProperty folder: String,
                                            @BeanProperty permission: String,
                                            @BeanProperty roles: String)

    val globalData = getGlobalPermissionData().map(data => {
      GlobalUserPermissionsViewRow(data.principal, data.userFullName, data.permission, data.roles.mkString(", "))
    }).toList
      .map(container => container.copy(permission = PermissionLabels.getLabel(container.permission)))
      .sortBy(containerPermissionAndPrincipal => (containerPermissionAndPrincipal.principal.toLowerCase, containerPermissionAndPrincipal.permission.toLowerCase))
      .slice(0, 11)

    val folderData = getFolderPermissionData(getRolesByName, Option(reportDefinition.filters).map(_.asScala.toSeq), None).map(data => {
      FolderUserPermissionsViewRow(data.principal, data.userFullName, data.folderName, data.permission, data.links.mkString(", "))
    }).toList
      .map(container => container.copy(permission = PermissionLabels.getLabel(container.permission)))
      .sortBy(containerPermissionAndPrincipal => (containerPermissionAndPrincipal.principal.toLowerCase, containerPermissionAndPrincipal.folder.toLowerCase, containerPermissionAndPrincipal.permission.toLowerCase))
      .slice(0, 11)

    Map("globalUserPermissionsData" -> globalData.asJava, "folderUserPermissionsData" -> folderData.asJava).asJava

  }

}
