package com.xebialabs.xlrelease.reports.excel

// scalastyle:off illegal.imports
import java.awt.Color
// scalastyle:on illegal.imports

import com.xebialabs.xlrelease.domain.status.FlagStatus
import com.xebialabs.xlrelease.reports.audit.CommonFormat
import org.apache.poi.ss.usermodel._
import org.apache.poi.xssf.usermodel.{XSSFCellStyle, XSSFColor, XSSFFont, XSSFWorkbook}

import scala.collection.mutable

object ExcelStyles {

  val LIGHT_BLUE: Color = new Color(95, 203, 244)
  val RED: Color = new Color(217, 76, 61)
  val GRAY: Color = new Color(128, 128, 128)
  val ORANGE: Color = new Color(253, 141, 16)
  val GREEN: Color = new Color(73,133,0)
  val LIGHT_GRAY: Color = new Color(213, 213, 213)
  val YELLOW: Color = Color.yellow

  val XL_GREEN: XSSFColor = new XSSFColor(new Color(73, 133, 0))

  def getColor(flagStatus: FlagStatus): Color = flagStatus match {
    case FlagStatus.ATTENTION_NEEDED => ORANGE
    case FlagStatus.AT_RISK => RED
    case _ => null
  }

}

class ExcelStyles(workbook: XSSFWorkbook) {

  val noBorderByColor = new mutable.HashMap[(Color, Color), XSSFCellStyle]()
  val noBorderDateByColor = new mutable.HashMap[(Color, Color), XSSFCellStyle]()

  lazy val default: XSSFCellStyle = {
    val style = workbook.createCellStyle()
    style.setFont(fontWithDefaultSize)
    style.setAlignment(HorizontalAlignment.LEFT)
    style.setVerticalAlignment(VerticalAlignment.TOP)
    style.setShrinkToFit(false)
    style
  }

  lazy val defaultStyle = default

  lazy val noGrid: XSSFCellStyle = {
    val style = workbook.createCellStyle()
    style.setFillForegroundColor(IndexedColors.WHITE.index)
    style.setFillPattern(FillPatternType.SOLID_FOREGROUND)
    style.setBorderLeft(BorderStyle.NONE)
    style.setBorderTop(BorderStyle.NONE)
    style.setBorderRight(BorderStyle.NONE)
    style.setBorderBottom(BorderStyle.NONE)
    style.setWrapText(false)
    style.setFont(fontWithDefaultSize)
    style.setVerticalAlignment(VerticalAlignment.TOP)
    style.setAlignment(HorizontalAlignment.LEFT)
    style
  }

  lazy val noGridBoldText: XSSFCellStyle = {
    val style = workbook.createCellStyle()
    style.cloneStyleFrom(noGrid)
    style.setFont(boldFont)
    style
  }

  lazy val noGridDateOnlyDate: XSSFCellStyle = {
    val style = workbook.createCellStyle()
    style.cloneStyleFrom(noGrid)
    style.setDataFormat(dateFormat)
    style.setAlignment(HorizontalAlignment.LEFT)
    style
  }

  lazy val noGridDate: XSSFCellStyle = {
    val style = workbook.createCellStyle()
    style.cloneStyleFrom(noGrid)
    style.setDataFormat(dateFormat)
    style.setAlignment(HorizontalAlignment.LEFT)
    style.setVerticalAlignment(VerticalAlignment.TOP)
    style
  }

  lazy val noGridNumeric: XSSFCellStyle = {
    val style = workbook.createCellStyle()
    style.cloneStyleFrom(noGrid)
    style.setAlignment(HorizontalAlignment.LEFT)
    style
  }

  lazy val whiteOnGreen: XSSFCellStyle = {
    val font: XSSFFont = workbook.createFont()
    font.setColor(IndexedColors.WHITE.index)
    font.setFontHeightInPoints(12)

    val style: XSSFCellStyle = workbook.createCellStyle()
    style.setFillForegroundColor(ExcelStyles.XL_GREEN)
    style.setFillPattern(FillPatternType.SOLID_FOREGROUND)
    style.setFont(font)
    style.setVerticalAlignment(VerticalAlignment.TOP)
    style.setBorderLeft(BorderStyle.NONE)
    style.setBorderTop(BorderStyle.NONE)
    style.setBorderRight(BorderStyle.NONE)
    style.setBorderBottom(BorderStyle.NONE)

    style
  }

  lazy val whiteOnGreenBig: XSSFCellStyle = {
    whiteOnGreen
  }

  lazy val leftAligned: XSSFCellStyle = {
    val leftAligned = workbook.createCellStyle()
    leftAligned.cloneStyleFrom(default)
    leftAligned.setAlignment(HorizontalAlignment.LEFT)
    leftAligned
  }

  lazy val centerAligned: XSSFCellStyle = {
    val leftAligned = workbook.createCellStyle()
    leftAligned.cloneStyleFrom(default)
    leftAligned.setAlignment(HorizontalAlignment.CENTER)
    leftAligned
  }

  lazy val rightAlignedNoGrid: XSSFCellStyle = {
    val rightAligned = workbook.createCellStyle()
    rightAligned.cloneStyleFrom(noGrid)
    rightAligned.setAlignment(HorizontalAlignment.RIGHT)
    rightAligned
  }

  lazy val leftAlignedNoGrid: XSSFCellStyle = {
    val rightAligned = workbook.createCellStyle()
    rightAligned.cloneStyleFrom(noGrid)
    rightAligned.setAlignment(HorizontalAlignment.LEFT)
    rightAligned
  }

  lazy val rightAligned: XSSFCellStyle = {
    val rightAligned = workbook.createCellStyle()
    rightAligned.cloneStyleFrom(default)
    rightAligned.setAlignment(HorizontalAlignment.RIGHT)
    rightAligned
  }

  lazy val rightAlignedBold: XSSFCellStyle = {
    val style = workbook.createCellStyle()
    style.cloneStyleFrom(rightAlignedNoGrid)
    val font = workbook.createFont
    font.setBold(true)
    style.setFont(font)
    style
  }

  lazy val wrapped: XSSFCellStyle = {
    val wrapped = workbook.createCellStyle()
    wrapped.cloneStyleFrom(default)
    wrapped.setWrapText(true)
    wrapped
  }

  lazy val wrappedDate: XSSFCellStyle = {
    val wrapped = workbook.createCellStyle()
    wrapped.cloneStyleFrom(default)
    wrapped.setWrapText(true)
    wrapped.setDataFormat(dateFormat)
    wrapped
  }

  lazy val hyperlink: XSSFCellStyle = {
    val style = workbook.createCellStyle
    val font = workbook.createFont
    font.setUnderline(FontUnderline.SINGLE)
    font.setColor(IndexedColors.BLUE.getIndex)
    font.setFontHeight(12)
    style.setFont(font)
    style
  }

  protected def increaseStyleFontSize(cellStyle: XSSFCellStyle, fontSize: Short): XSSFCellStyle = {
    val whiteOnGreenWithBiggerFont = workbook.createCellStyle()
    whiteOnGreenWithBiggerFont.cloneStyleFrom(cellStyle)
    val increasedFont = workbook.createFont()
    increasedFont.setColor(cellStyle.getFont.getColor)
    increasedFont.setFontHeightInPoints(fontSize)
    increasedFont.setBold(cellStyle.getFont.getBold)
    whiteOnGreenWithBiggerFont.setFont(increasedFont)
    whiteOnGreenWithBiggerFont
  }

  lazy val boldFont: XSSFFont = {
    val font = workbook.createFont()
    font.setBold(true)
    font.setFontHeightInPoints(12)
    font
  }

  lazy val italicFont: XSSFFont = {
    val font = workbook.createFont()
    font.setItalic(true)
    font.setFontHeightInPoints(12)
    font
  }

  lazy val errorFont: XSSFFont = {
    val font = copyFont(italicFont)
    font.setColor(new XSSFColor(ExcelStyles.GRAY))
    font
  }

  lazy val date: XSSFCellStyle = {
    val style = workbook.createCellStyle()
    style.setDataFormat(dateFormat)
    style.setVerticalAlignment(VerticalAlignment.TOP)
    style.setFont(fontWithDefaultSize)
    style.setAlignment(HorizontalAlignment.LEFT)
    style
  }

  lazy val bold: XSSFCellStyle = {
    val style = workbook.createCellStyle()
    style.setFont(boldFont)
    style
  }

  lazy val error: XSSFCellStyle = {
    val style = workbook.createCellStyle()
    style.cloneStyleFrom(default)
    style.setFont(errorFont)
    style
  }

  def noBorder(color: Color = null, backgroundColor: Color = null): XSSFCellStyle = {
    noBorderByColor.getOrElseUpdate(
      (color, backgroundColor),
      addBackgroundColor(
        addFontColor(noGrid, color),
        backgroundColor)
    )
  }

  def noBorderDate(color: Color = null, backgroundColor: Color = null): XSSFCellStyle = {
    noBorderDateByColor.getOrElseUpdate(
      (color, backgroundColor),
      addBackgroundColor(
        addFontColor(noGridDate, color),
        backgroundColor)
    )
  }

  private lazy val dateFormat: Short = workbook.getCreationHelper.createDataFormat.getFormat(CommonFormat.DATE_FORMAT)

  private lazy val fontWithDefaultSize: XSSFFont = {
    val font = workbook.createFont()
    font.setFontHeightInPoints(12)
    font
  }

  private def addFontColor(styleToCopy: XSSFCellStyle, color: Color): XSSFCellStyle = {
    if (color != null) {
      val style = workbook.createCellStyle
      style.cloneStyleFrom(styleToCopy)
      val font = copyFont(styleToCopy.getFont)
      font.setColor(new XSSFColor(color))
      style.setFont(font)
      style
    } else {
      styleToCopy
    }
  }

  def addBackgroundColor(styleToCopy: XSSFCellStyle, color: Color): XSSFCellStyle = {
    if (color != null) {
      val style = workbook.createCellStyle
      style.cloneStyleFrom(styleToCopy)
      style.setFillForegroundColor(new XSSFColor(color))
      style.setFillPattern(FillPatternType.SOLID_FOREGROUND)
      style
    } else {
      styleToCopy
    }
  }

  private def copyFont(font: XSSFFont) = {
    val copy = workbook.createFont
    copy.setBold(font.getBold)
    copy.setColor(font.getColor)
    copy.setItalic(font.getItalic)
    copy.setUnderline(font.getUnderline)
    copy.setCharSet(font.getCharSet)
    copy.setFamily(font.getFamily)
    copy.setFontHeight(font.getFontHeight)
    copy.setFontName(font.getFontName)
    copy.setStrikeout(font.getStrikeout)
    copy.setTypeOffset(font.getTypeOffset)
    copy
  }

}
