package com.xebialabs.xlrelease.reports.upgrade


import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, Type}
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.db.ArchivedReleases
import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.facet.TaskReportingRecord
import com.xebialabs.xlrelease.reports.repository.build.BuildRecordRepository
import com.xebialabs.xlrelease.reports.repository.codecompliance.CodeComplianceRecordRepository
import com.xebialabs.xlrelease.reports.repository.deployment.DeploymentRecordRepository
import com.xebialabs.xlrelease.reports.repository.itsm.ItsmRecordRepository
import com.xebialabs.xlrelease.reports.repository.plan.PlanRecordRepository
import com.xebialabs.xlrelease.reports.upgrade.XLRelease950ArchiveFacetUpgrade.{PLUGIN_TASK_TYPE_PREFIXES, XLR900_START_DATE}
import com.xebialabs.xlrelease.repository.FacetRepositoryDispatcher
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.UpgradeSupport.{BatchSupport, TransactionSupport}
import grizzled.slf4j.Logging
import org.codehaus.jettison.json.JSONObject
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component
import org.springframework.transaction.support.TransactionTemplate

import java.io.ByteArrayInputStream
import java.nio.charset.StandardCharsets
import java.sql.PreparedStatement
import java.text.SimpleDateFormat
import java.util.Date
import scala.jdk.CollectionConverters._

object XLRelease950ArchiveFacetUpgrade {
  val XLR900_START_DATE: Date = new SimpleDateFormat("yyyy-MM-dd").parse("2019-07-01");
  val PLUGIN_TASK_TYPE_PREFIXES: Set[String] = Set("servicenow", "jira", "jenkins")
}

@Component
class XLRelease950ArchiveFacetUpgrade @Autowired()(@Qualifier("reportingJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                   @Qualifier("reportingSqlDialect") val dialect: Dialect,
                                                   @Qualifier("reportTransactionTemplate") val transactionTemplate: TransactionTemplate,
                                                   archivedReleases: ArchivedReleases,
                                                   sqlRepositoryAdapter: SqlRepositoryAdapter,
                                                   facetRepositoryDispatcher: FacetRepositoryDispatcher,
                                                   buildRecordRepository: BuildRecordRepository,
                                                   planRecordRepository: PlanRecordRepository,
                                                   itsmRecordRepository: ItsmRecordRepository,
                                                   codeComplianceRecordRepository: CodeComplianceRecordRepository,
                                                   deploymentRecordRepository: DeploymentRecordRepository)
  extends Upgrade
    with PersistenceSupport
    with Logging
    with BatchSupport
    with TransactionSupport {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "9.5.0#2")

  override def doUpgrade(): Boolean = {
    val supportedFacetTypes = facetRepositoryDispatcher.supportedTypes ++ deprecatedFacetTypes

    PLUGIN_TASK_TYPE_PREFIXES.foreach { prefix =>
      val taskTypes = getPluginTaskTypes(prefix)
      val releaseIds = getReleasesIds(taskTypes)
      doInBatch(releaseIds, itemsName = prefix) { batch =>
        batch.items.foreach { releaseId =>
          archivedReleases.getRelease(releaseId).foreach { releaseString =>
            val releaseJson = new JSONObject(releaseString)
            val phases = releaseJson.getJSONArray("phases")
            doInTransaction {
              for {
                phaseIdx <- 0 until phases.length()
                phase = phases.getJSONObject(phaseIdx)
                tasks = phase.getJSONArray("tasks")
                taskIdx <- 0 until tasks.length()
                if tasks.getJSONObject(taskIdx).has("facets")
                facets = tasks.getJSONObject(taskIdx).getJSONArray("facets")
                facetsSize = facets.length()
                i <- 0 until facets.length()
                facetType = Type.valueOf(facets.getJSONObject(i - (facetsSize - facets.length())).getString("type"))
                if supportedFacetTypes.exists(facetType.instanceOf)
              } yield {
                val facetJson = facets.getJSONObject(i - (facetsSize - facets.length()))
                val facetObj = deserializeFacet[TaskReportingRecord](facetJson)
                facetObj.foreach { facet =>
                  logger.trace(s"processing facet ${facet.getId}/${facet.getType.toString}")
                  setDefaultValuesForRequiredProperties(facet)
                  renewDeprecatedTypes(facet)
                  try {
                    facetRepositoryDispatcher.archiveRepository.create(facet)
                  } catch {
                    case e: NotFoundException => logger.warn(e.getMessage)
                  }
                }
                facets.remove(facetJson)
              }
              updateRelease(releaseJson.getString("id"), releaseJson.toString())
            }
          }
        }
      }
    }

    true
  }

  def getPluginTaskTypes(prefix: String): List[String] = DescriptorRegistry.getDescriptors.asScala
    .filter(pd => prefix.equals(pd.getType.getPrefix)).map(_.getType.toString).toList

  private def getReleasesIds(taskTypes: List[String]): Seq[String] =
    taskTypes match {
      case Nil => Seq.empty
      case _ => sqlQuery(
        s""" SELECT r.$REPORT_RELEASES_ID_COLUMN
           | FROM $REPORT_RELEASES_TABLE_NAME r
           | LEFT JOIN $REPORT_TASKS_TABLE_NAME t
           | ON r.$REPORT_RELEASES_ID_COLUMN = t.$REPORT_TASKS_RELEASEID_COLUMN
           | WHERE r.$REPORT_RELEASES_START_DATE_COLUMN >= :xlr900StartDate
           | AND t.$REPORT_TASKS_TASK_TYPE_COLUMN IN (:taskTypesWithFacets) """.stripMargin,
        params("xlr900StartDate" -> XLR900_START_DATE, "taskTypesWithFacets" -> taskTypes.asJava),
        _.getString(REPORT_RELEASES_ID_COLUMN)).toSeq
    }

  private def deserializeFacet[T <: TaskReportingRecord](facetJson: JSONObject): Option[T] = {
    val facetOption = sqlRepositoryAdapter.deserialize[T](facetJson.toString)
    facetOption.foreach(facet => facet.setTargetId(facetJson.getString("targetId")))
    facetOption
  }

  private def updateRelease(releaseId: String, releaseJson: String): Boolean = {
    jdbcTemplate.update(
      s"""|UPDATE $REPORT_RELEASES_TABLE_NAME
          | SET
          |   $REPORT_RELEASES_RELEASEJSON_COLUMN = ?
          | WHERE $REPORT_RELEASES_ID_COLUMN = ?""".stripMargin,
      (ps: PreparedStatement) => {
        ps.setBinaryStream(1, new ByteArrayInputStream(releaseJson.getBytes(StandardCharsets.UTF_8)))
        ps.setString(2, shortenId(releaseId))
      }
    ) == 1
  }
}
