package com.xebialabs.xlrelease.reports.api.internal;

import java.time.Instant;
import java.util.*;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Controller;
import com.google.common.collect.Maps;

import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.xlrelease.api.utils.ResponseHelper;
import com.xebialabs.xlrelease.api.v1.forms.TimeFrame;
import com.xebialabs.xlrelease.domain.UserProfile;
import com.xebialabs.xlrelease.domain.validators.UserAccountValidator;
import com.xebialabs.xlrelease.reports.dto.*;
import com.xebialabs.xlrelease.reports.job.api.ReportDefinition;
import com.xebialabs.xlrelease.reports.job.api.ReportingEngineService;
import com.xebialabs.xlrelease.reports.job.api.StreamingReportResult;
import com.xebialabs.xlrelease.reports.job.domain.ReportJob;
import com.xebialabs.xlrelease.reports.job.domain.ReportJobStatus;
import com.xebialabs.xlrelease.reports.job.repository.ReportJobFilters;
import com.xebialabs.xlrelease.reports.pdf.DashboardReportService;
import com.xebialabs.xlrelease.reports.service.ReportParams;
import com.xebialabs.xlrelease.reports.service.ReportServiceCache;
import com.xebialabs.xlrelease.reports.service.ReportsService;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.XLReleasePermissions;
import com.xebialabs.xlrelease.service.UserProfileService;
import com.xebialabs.xlrelease.utils.SortSupport;
import com.xebialabs.xlrelease.views.Point;

import io.micrometer.core.annotation.Timed;

import static com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.xlrelease.api.ApiService.DEFAULT_RESULTS_PER_PAGE;
import static com.xebialabs.xlrelease.api.ApiService.PAGE;
import static com.xebialabs.xlrelease.api.ApiService.RESULTS_PER_PAGE;
import static com.xebialabs.xlrelease.reports.utils.ReportUtils.splitTags;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.AUDIT_ALL;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.VIEW_REPORTS;
import static scala.jdk.javaapi.CollectionConverters.asJava;

/**
 * Provides data to the various reports available in the Digital.ai Release UI.
 */
@Path("/reports")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class ReportResource {

    private PermissionChecker permissions;
    private ReportServiceCache reportService;
    private ReportingEngineService reportingEngineService;
    private UserProfileService userProfileService;
    private ReportsService reportsService;
    private DashboardReportService dashboardReportService;

    @Autowired
    public ReportResource(PermissionChecker permissions, ReportServiceCache reportService, ReportingEngineService reportingEngineService,
                          UserProfileService userProfileService, ReportsService reportsService, DashboardReportService dashboardReportService) {
        this.permissions = permissions;
        this.reportService = reportService;
        this.reportingEngineService = reportingEngineService;
        this.userProfileService = userProfileService;
        this.reportsService = reportsService;
        this.dashboardReportService = dashboardReportService;
    }

    @POST
    @Timed
    @Path("releases/duration")
    public ReleasesDuration getReleaseDuration(ReportForm reportForm) {
        checkPermission(reportForm);
        return reportService.getReleaseDuration(ReportParams.apply(reportForm));
    }

    @POST
    @Timed
    @Path("most-involved-people")
    public List<UserParticipation> getTopPeopleMostInvolved(ReportForm reportForm) {
        checkPermission(reportForm);
        return reportService.getTopPeopleMostInvolved(ReportParams.apply(reportForm));
    }

    @POST
    @Timed
    @Path("longest-tasks")
    public List<TaskDuration> getTopLongestTasks(ReportForm reportForm) {
        checkPermission(reportForm);
        return reportService.getTopLongestTasks(ReportParams.apply(reportForm));
    }

    @POST
    @Timed
    @Path("longest-task-types")
    public List<LongestTaskType> getTopLongestTaskTypes(ReportForm reportForm) {
        checkPermission(reportForm);
        return reportService.getTopLongestTaskTypes(ReportParams.apply(reportForm));
    }

    @POST
    @Timed
    @Path("average-and-longest-task-duration")
    public AverageAndLongestTaskDuration getAverageAndLongestTaskDuration(ReportForm reportForm) {
        checkPermission(reportForm);
        return reportService.getAverageAndLongestTaskDuration(ReportParams.apply(reportForm));
    }

    @POST
    @Timed
    @Path("release/completed")
    public CompletedReleases getCompletedReleases(ReportForm reportForm) {
        checkPermission(reportForm);
        return reportService.getCompletedReleases(ReportParams.apply(reportForm));
    }

    @POST
    @Timed
    @Path("release/average-and-longest-release-duration")
    public AverageAndLongestReleaseDuration getAverageAndLongestReleaseDuration(ReportForm reportForm) {
        checkPermission(reportForm);
        return reportService.getAverageAndLongestReleaseDuration(ReportParams.apply(reportForm));
    }

    @POST
    @Timed
    @Path("releases/automation")
    public ReleasesAutomation getReleaseAutomation(ReportForm reportForm) {
        checkPermission(reportForm);
        return reportService.getReleaseAutomationSeries(ReportParams.apply(reportForm));
    }

    @POST
    @Timed
    @Path("longest-phases")
    public List<PhaseDuration> getTopLongestPhases(ReportForm reportForm) {
        checkPermission(reportForm);
        return reportService.getTopLongestPhases(ReportParams.apply(reportForm));
    }

    @POST
    @Timed
    @Path("releases/number-by-month")
    public List<Point> getNumberOfReleaseByMonth(ReportForm reportForm) {
        checkPermission(reportForm);
        return reportService.getNumberOfReleaseByMonth(ReportParams.apply(reportForm));
    }

    @POST
    @Timed
    @Path("releases/most-recent/{limit}")
    public List<ReleaseAutomationData> getMostRecentReleaseData(@PathParam("limit") Integer limit, ReportForm reportForm) {
        checkPermission(reportForm);
        ReportParams params = ReportParams.apply(
                TimeFrame.RANGE, reportForm.getFrom(), reportForm.getTo(), splitTags(reportForm.getTags()),
                reportForm.getFilters(), reportForm.isUserSpecific(), limit, reportForm.isRefresh()
        );
        return reportService.getMostRecentReleasesData(params);
    }

    private void checkPermission(ReportForm reportForm) {
        if (!permissions.hasGlobalPermission(AUDIT_ALL) && reportForm.isUserSpecific()) {
            permissions.check(VIEW_REPORTS);
        }
    }

    @POST
    @Timed
    @Path("preview")
    @Produces(MediaType.APPLICATION_JSON)
    public Response preview(ReportDefinition report) {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        String generatedBy = Permissions.getAuthenticatedUserName();
        report.setGeneratedBy(generatedBy);
        Map<String, Object> response = (Map<String, Object>) reportingEngineService.preview(report);
        return Response.ok(response).build();
    }

    @POST
    @Timed
    @Path("submit")
    @Produces(MediaType.APPLICATION_JSON)
    public Response submit(ReportDefinition report) {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        String generatedBy = Permissions.getAuthenticatedUserName();
        report.setGeneratedBy(generatedBy);
        report.setGeneratedOn(Date.from(Instant.now()));
        Integer jobId = reportingEngineService.submit(report);
        Map<String, Object> submitResult = Maps.newHashMap();
        submitResult.put("jobId", jobId);
        UserProfile userProfile = userProfileService.findByUsername(generatedBy);
        if (isValidEmailConfigured(userProfile.getEmail())) {
            submitResult.put("emailConfigured", true);
        } else {
            submitResult.put("emailConfigured", false);
        }
        return Response.ok(submitResult).build();
    }

    @GET
    @Timed
    @Path("download/{jobId}")
    @Produces(MediaType.APPLICATION_OCTET_STREAM)
    public Response download(@PathParam("jobId") Integer jobId) {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        final StreamingReportResult reportResult = reportingEngineService.getResult(jobId);
        return ResponseHelper.streamFile(reportResult.fileName(), reportResult::write, reportResult.contentType());
    }

    @GET
    @Timed
    @Path("status/{jobId}")
    @Produces(MediaType.APPLICATION_JSON)
    public Response status(@PathParam("jobId") Integer jobId) {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        ReportJobStatus status = reportingEngineService.status(jobId);
        return Response.ok(status).build();
    }

    @GET
    @Timed
    @Path("{jobId}")
    @Produces(MediaType.APPLICATION_JSON)
    public Response get(@PathParam("jobId") Integer jobId) {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        ReportJob reportJob = reportingEngineService.get(jobId);
        return Response.ok(reportJob).build();
    }

    @GET
    @Timed
    @Path("search")
    @Produces(MediaType.APPLICATION_JSON)
    public Page<ReportJob> search(@BeanParam ReportJobFilters reportJobFilters,
                                  @DefaultValue("0") @QueryParam(PAGE) int page,
                                  @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) int resultsPerPage,
                                  @Context UriInfo uriInfo) {
        checkArgument(resultsPerPage <= DEFAULT_RESULTS_PER_PAGE, "Number of results per page cannot be more than 100");
        PageRequest pageable = PageRequest.of(page, resultsPerPage, SortSupport.toSort(uriInfo));
        return search(reportJobFilters, pageable);
    }

    @POST
    @Timed
    @Path("poll")
    @Produces(MediaType.APPLICATION_JSON)
    public List<ReportJob> poll(List<Integer> jobIds) {
        checkNotNull(jobIds, "Report job id list can not be null.", "");
        if (jobIds.isEmpty()) {
            return Collections.emptyList();
        } else {
            ReportJobFilters reportJobFilters = new ReportJobFilters();
            reportJobFilters.setJobIds(jobIds);
            PageRequest pageable = PageRequest.of(0, Integer.MAX_VALUE);
            return search(reportJobFilters, pageable).getContent();
        }
    }

    @GET
    @Timed
    @Path("applicationNames")
    @Produces(MediaType.APPLICATION_JSON)
    public Set<String> applicationsNames() {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        return asJava(reportsService.getAllApplicationNames());
    }

    @GET
    @Timed
    @Path("environmentNames")
    @Produces(MediaType.APPLICATION_JSON)
    public Set<String> environmentNames() {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        return asJava(reportsService.getAllEnvironmentNames());
    }

    @POST
    @Timed
    @Path("{jobId}/abort")
    @Produces(MediaType.APPLICATION_JSON)
    public Response abort(@PathParam("jobId") Integer jobId) {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        reportingEngineService.abort(jobId);
        return Response.ok().build();
    }

    @POST
    @Timed
    @Path("abort")
    @Produces(MediaType.APPLICATION_JSON)
    public Response abort(List<Integer> jobIds) {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        checkNotNull(jobIds, "Report job id list can not be null.", "");

        List<Integer> abortedJobIds = reportingEngineService.abort(jobIds);
        return returnList("abortedJobIds", abortedJobIds);
    }

    @POST
    @Timed
    @Path("abort/all")
    @Produces(MediaType.APPLICATION_JSON)
    public Response abortAll() {
        permissions.check(XLReleasePermissions.AUDIT_ALL);

        List<Integer> abortedJobIds = reportingEngineService.abortAll();
        return returnList("abortedJobIds", abortedJobIds);
    }


    @DELETE
    @Timed
    @Path("{jobId}")
    @Produces(MediaType.APPLICATION_JSON)
    public Response delete(@PathParam("jobId") Integer jobId) {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        reportingEngineService.delete(jobId);
        return Response.ok().build();
    }

    @POST
    @Timed
    @Path("delete")
    @Produces(MediaType.APPLICATION_JSON)
    public Response delete(List<Integer> jobIds) {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        checkNotNull(jobIds, "Report job id list can not be null.", "");
        List<Integer> deletedJobIds = reportingEngineService.delete(jobIds);
        return returnList("deletedJobIds", deletedJobIds);
    }

    @POST
    @Timed
    @Path("delete/all")
    @Produces(MediaType.APPLICATION_JSON)
    public Response deleteAll() {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        List<Integer> deletedJobIds = reportingEngineService.deleteAll();
        return returnList("deletedJobIds", deletedJobIds);
    }

    @POST
    @Path("pdf")
    @Produces(MediaType.APPLICATION_OCTET_STREAM)
    public Response generateReportPdf(@QueryParam("reportName") String reportName, String pngContent) {
        StreamingOutput output = outputStream -> dashboardReportService.generateDashboardReport(pngContent, reportName, outputStream);
        return ResponseHelper.streamFile(reportName + ".pdf", output, "application/pdf");
    }

    private Response returnList(String listTitle, List<Integer> items) {
        return Response.ok(Collections.singletonMap(listTitle, items)).build();
    }

    private Page<ReportJob> search(ReportJobFilters reportJobFilters, Pageable pageable) {
        permissions.check(XLReleasePermissions.AUDIT_ALL);
        ReportJobFilters filters = reportJobFilters == null ? new ReportJobFilters() : reportJobFilters;
        return reportingEngineService.findBy(filters, pageable);
    }

    private boolean isValidEmailConfigured(final String email) {
        return isNotBlank(email) && UserAccountValidator.isValidEmailAddress(email);
    }

    public enum ReportType {
        RELEASE_DURATION,
        RELEASE_AUTOMATION,
        TOP_PEOPLE_MOST_INVOLVED,
        TOP_LONGEST_TASK,
        TOP_LONGEST_PHASE,
        TOP_LONGEST_RELEASE,
        NUMBER_OF_RELEASE_BY_MONTH
    }
}
