package com.xebialabs.xlrelease.reports.excel

// scalastyle:off illegal.imports
import com.xebialabs.xlrelease.reports.domain.ReportSheetError
import org.apache.poi.ss.util.CellReference

import java.awt.Color
// scalastyle:on illegal.imports

import com.xebialabs.xlrelease.reports.domain.ReportCellError
import com.xebialabs.xlrelease.reports.excel.ErrorSheet._
import org.apache.commons.lang.StringUtils
import org.apache.poi.common.usermodel.HyperlinkType
import org.apache.poi.ss.usermodel.CellType
import org.apache.poi.ss.util.CellRangeAddress

import scala.jdk.CollectionConverters._

object ErrorSheet {
  val SHEET_COLUMN = ExcelHeaderColumn("Sheet", 30)
  val CELL_COLUMN = ExcelHeaderColumn("Cell", 30)
  val ERROR_COLUMN = ExcelHeaderColumn("Error", 30)
}

class ErrorSheet(val sheetWriter: ExcelSheetWriter) {

  def addContent(styles: ExcelStyles): Unit = {
    addHeader(styles)

    sheetWriter.getReportWorkbook.getSheetErrors.asScala.foreach(e => addSheetError(e))
    sheetWriter.getReportWorkbook.getCellErrors.asScala.foreach(e => addError(e))
  }

  private def addHeader(styles: ExcelStyles) = {
    sheetWriter.newRow()
      .addCell("Note, that links may point to wrong cell if you make any changes", styles.addBackgroundColor(styles.bold, Color.YELLOW))
      .mergeCells(new CellRangeAddress(sheetWriter.rowIndex - 1, sheetWriter.rowIndex - 1, sheetWriter.columnIndex - 1, sheetWriter.columnIndex))
      .newRow()
      .addCell("(for example change order of rows in tables by using sort feature)", styles.addBackgroundColor(styles.bold, Color.YELLOW))
      .mergeCells(new CellRangeAddress(sheetWriter.rowIndex - 1, sheetWriter.rowIndex - 1, sheetWriter.columnIndex - 1, sheetWriter.columnIndex))
      .newRow()
      .newRow()
      .addHeaderCell(SHEET_COLUMN, styles.whiteOnGreen)
      .addHeaderCell(CELL_COLUMN, styles.whiteOnGreen)
      .addHeaderCell(ERROR_COLUMN, styles.whiteOnGreen)
  }

  protected def addError(reportError: ReportCellError): Unit = {
    sheetWriter.newRow()

    val humanReadableAddressOfCellWithData = s"${CellReference.convertNumToColString(reportError.col)}${reportError.row}"
    val addressOfCellWithData = s"'${reportError.sheet.getSheetName}'!$humanReadableAddressOfCellWithData"

    val linkFromErrorSheetToReportSheetForSheetNameCell = sheetWriter.getWorkbook.getCreationHelper.createHyperlink(HyperlinkType.DOCUMENT)
    linkFromErrorSheetToReportSheetForSheetNameCell
      .setAddress(addressOfCellWithData)
    val linkFromErrorSheetToReportSheetForAddressCell = sheetWriter.getWorkbook.getCreationHelper.createHyperlink(HyperlinkType.DOCUMENT)
    linkFromErrorSheetToReportSheetForAddressCell
      .setAddress(addressOfCellWithData)

    sheetWriter.addCell(reportError.sheet.getSheetName)
    sheetWriter.getSheet.getRow(sheetWriter.rowIndex - 1).getCell(sheetWriter.columnIndex - 1)
      .setHyperlink(linkFromErrorSheetToReportSheetForSheetNameCell)
    sheetWriter.addCell(humanReadableAddressOfCellWithData)
    sheetWriter.getSheet.getRow(sheetWriter.rowIndex - 1).getCell(sheetWriter.columnIndex - 1)
      .setHyperlink(linkFromErrorSheetToReportSheetForAddressCell)
    sheetWriter.addCell(reportError.throwable.getMessage)

    val addressOfCellWithErrorDescription = s"'${sheetWriter.getSheet.getSheetName}'!${CellReference.convertNumToColString(sheetWriter.columnIndex - 1)}${sheetWriter.rowIndex}"

    val linkFromReportSheetToErrorSheet = sheetWriter.getWorkbook.getCreationHelper.createHyperlink(HyperlinkType.DOCUMENT)
    linkFromReportSheetToErrorSheet
      .setAddress(addressOfCellWithErrorDescription)

    val targetCell = reportError.sheet.getRow(reportError.row - 1).getCell(reportError.col)

    targetCell.setHyperlink(linkFromReportSheetToErrorSheet)
    if (targetCell.getCellType == CellType.STRING && (targetCell.getStringCellValue == null || targetCell.getStringCellValue.isEmpty)) {
      targetCell.setCellValue(StringUtils.repeat("\u00A0", 255))
    }
  }

  protected def addSheetError(reportSheetError: ReportSheetError): Unit = {
    sheetWriter.newRow()
    sheetWriter.addCell(reportSheetError.sheetName)
    sheetWriter.addCell("")
    sheetWriter.addCell(reportSheetError.throwable.getMessage)
  }

}
