package com.xebialabs.xlrelease.reports.repository.plan

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.udm.reporting.PlanRecord
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.reports.repository.CommonRowAndParamsMapper._
import com.xebialabs.xlrelease.reports.repository.plan.PlanRecordLiveRepository._
import com.xebialabs.xlrelease.reports.repository.{RepositoryExceptionUtils, SearchRecordsSupport}
import com.xebialabs.xlrelease.repository.FacetRepository.SpecializedFacetRepository
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, RELEASES, TASKS}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, TaskCiUid, TaskPersistence}
import com.xebialabs.xlrelease.repository.{FacetRepository, Ids}
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.utils.FolderId
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.collection.mutable

object PlanRecordLiveRepository {

  import PLAN_TASK_REPORTING_RECORD_LIVE._

  private lazy val ALL_FIELDS =
    s"""|${commonFieldNames(PLAN_TASK_REPORTING_RECORD_LIVE, "p")},
       |p.$TASK_UID,
       |p.$TICKET,
       |p.$TITLE,
       |p.$TICKET_TYPE,
       |p.$STATUS,
       |p.$UPDATED_DATE,
       |p.$UPDATED_BY,
       |p.$TICKET_URL,
       |f.${FOLDERS.FOLDER_PATH},
       |f.${FOLDERS.FOLDER_ID},
       |r.${RELEASES.RELEASE_ID},
       |t.${TASKS.TASK_ID}""".stripMargin

  private lazy val SELECT_ALL_PLAN_RECORDS: String =
    s"""|SELECT
        | $ALL_FIELDS
        |FROM $TABLE p
        |JOIN ${TASKS.TABLE} t ON t.${TASKS.CI_UID} = p.$TASK_UID
        |JOIN ${RELEASES.TABLE} r ON r.${RELEASES.CI_UID} = t.${TASKS.RELEASE_UID}
        |LEFT OUTER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = r.${RELEASES.FOLDER_CI_UID}""".stripMargin

  private lazy val SELECT_PLAN_RECORD_BY_ID_STMT: String =
    s"""|$SELECT_ALL_PLAN_RECORDS
        |WHERE p.$RECORD_ID = :recordId""".stripMargin

  private lazy val SELECT_PLAN_RECORDS_BY_TASK_ID_STMT: String =
    s"""|$SELECT_ALL_PLAN_RECORDS
        |WHERE t.${TASKS.TASK_ID} = :taskId""".stripMargin

  private lazy val SELECT_PLAN_RECORDS_BY_RELEASE_ID_STMT: String =
    s"""|$SELECT_ALL_PLAN_RECORDS
        |WHERE r.${RELEASES.RELEASE_ID} = :releaseId""".stripMargin

  private lazy val EXISTS_PLAN_RECORD_BY_ID: String =
    s"""SELECT COUNT($RECORD_ID) FROM $TABLE WHERE $RECORD_ID = :recordId""".stripMargin

  private lazy val INSERT_PLAN_RECORD_STMT: String =
    s"""| INSERT INTO $TABLE (
       | ${commonFieldNames(PLAN_TASK_REPORTING_RECORD_LIVE)},
       | $TASK_UID,
       | $TICKET,
       | $TITLE,
       | $TICKET_TYPE,
       | $STATUS,
       | $UPDATED_DATE,
       | $UPDATED_BY,
       | $TICKET_URL
       |) VALUES (
       | ${commonParameterNames()},
       | :taskUid,
       | :ticket,
       | :title,
       | :ticketType,
       | :status,
       | :updatedDate,
       | :updatedBy,
       | :ticketUrl
       |)""".stripMargin
  private lazy val DELETE_PLAN_RECORD_STMT: String =
    s"""|DELETE FROM $TABLE
        |WHERE $RECORD_ID = :recordId""".stripMargin
}

class PlanRecordLiveRepository(taskPersistence: TaskPersistence,
                               val ciIdService: CiIdService,
                               val jdbcTemplate: JdbcTemplate,
                               val dialect: Dialect) extends SpecializedFacetRepository(Type.valueOf(classOf[PlanRecord]))
  with FacetRepository.ForLive
  with PersistenceSupport
  with SearchRecordsSupport[PlanRecord]
  with RepositoryExceptionUtils
  with Logging {

  override def get(recordId: String, recordType: Option[Type]): Facet = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(SELECT_PLAN_RECORD_BY_ID_STMT, params("recordId" -> getName(recordId.normalized)), planRowMapper)
      .headOption.getOrElse(throw new NotFoundException(s"Record '$recordId' not found."))
  }

  override protected def doCreate(facet: Facet): Facet = {
    logger.debug("Creating Plan record in live")
    requiresInstanceOfSupportedType(facet.getType)
    withTargetTaskUid(facet.getTargetId) { taskUid =>
      val planFacet = facet.asInstanceOf[PlanRecord]
      sqlInsert(INSERT_PLAN_RECORD_STMT, asQueryParameters(planFacet, taskUid))
      facet.setTargetId(facet.getTargetId)
      facet
    }.getOrElse {
      throw targetTaskNotFoundException(facet.getId, facet.getTargetId)
    }
  }

  override def delete(recordId: String, recordType: Option[Type]): Unit =
    sqlUpdate(DELETE_PLAN_RECORD_STMT, params("recordId" -> getName(recordId.normalized)), _ => ())

  override def update(facet: Facet): Facet = throw new UnsupportedOperationException("You cannot update records!")

  override def search(facetsFilters: FacetFilters): Seq[Facet] = searchRecords(facetsFilters, SELECT_ALL_PLAN_RECORDS, planRowMapper)

  override def exists(recordId: String, recordType: Option[Type]): Boolean = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(EXISTS_PLAN_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), _.getInt(1) > 0).headOption.exists(identity)
  }

  override def findAllFacetsByRelease(release: Release): Seq[Facet] = findAllFacetsByReleaseId(release.getId).toSeq

  override def findAllFacetsByTask(task: Task): Seq[Facet] = findAllFacetsByTargetId(task.getId).toSeq

  override protected def findAllFacetsByTargetId(targetId: CiId): mutable.Seq[PlanRecord] =
    sqlQuery(SELECT_PLAN_RECORDS_BY_TASK_ID_STMT, params("taskId" -> getFolderlessId(targetId)), planRowMapper)

  override protected def findAllFacetsByReleaseId(releaseId: CiId): mutable.Seq[PlanRecord] =
    sqlQuery(SELECT_PLAN_RECORDS_BY_RELEASE_ID_STMT, params("releaseId" -> getName(releaseId.normalized)), planRowMapper)

  private def withTargetTaskUid[A](targetId: CiId)(f: TaskCiUid => A): Option[A] =
    taskPersistence.getTaskUidById(targetId).map(f)

  private val planRowMapper: RowMapper[PlanRecord] = (rs: ResultSet, _: Int) => {
    val planRecord: PlanRecord = Type.valueOf(rs.getString(PLAN_TASK_REPORTING_RECORD_LIVE.RECORD_TYPE))
      .getDescriptor.newInstance[PlanRecord](rs.getString(PLAN_TASK_REPORTING_RECORD_LIVE.RECORD_ID))
    val folderPath = Option(rs.getString(FOLDERS.FOLDER_PATH)).map(FolderId.apply).getOrElse(FolderId.Root)
    val releaseId = rs.getString(RELEASES.RELEASE_ID)
    val taskId = Ids.getReleaselessChildId(rs.getString(TASKS.TASK_ID))
    val targetId = folderPath / rs.getString(FOLDERS.FOLDER_ID) / releaseId / taskId
    planItemRowMapper(rs, planRecord, targetId.absolute, PLAN_TASK_REPORTING_RECORD_LIVE)
  }
}
