package com.xebialabs.xlrelease.reports.repository.codecompliance

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters
import com.xebialabs.xlrelease.db.ArchivedReleases.shortenId
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.udm.reporting.CodeComplianceRecord
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.reports.repository.CommonRowAndParamsMapper._
import com.xebialabs.xlrelease.reports.repository.codecompliance.CodeComplianceRecordLiveRepository._
import com.xebialabs.xlrelease.reports.repository.{RepositoryExceptionUtils, SearchRecordsSupport}
import com.xebialabs.xlrelease.repository.FacetRepository.SpecializedFacetRepository
import com.xebialabs.xlrelease.repository.Ids.{getFolderlessId, getName}
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.{CiId, _}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, RELEASES, TASKS}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, TaskCiUid, TaskPersistence}
import com.xebialabs.xlrelease.repository.{FacetRepository, Ids}
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.utils.FolderId
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.collection.mutable

object CodeComplianceRecordLiveRepository {

  import CODE_COMPLIANCE_TASK_REPORTING_RECORD_LIVE._

  private lazy val alias = "c"
  private lazy val ALL_FIELDS =
    s"""
       | ${commonFieldNames(CODE_COMPLIANCE_TASK_REPORTING_RECORD_LIVE, alias)},
       | f.${FOLDERS.FOLDER_PATH},
       | f.${FOLDERS.FOLDER_ID},
       | r.${RELEASES.RELEASE_ID},
       | t.${TASKS.TASK_ID},
       | $alias.$SERVER_USER,
       | $alias.$PROJECT,
       | $alias.$PROJECT_URL,
       | $alias.$ANALYSIS_DATE,
       | $alias.$OUTCOME,
       | $alias.$COMPLIANCE_DATA
       |""".stripMargin

  private lazy val SELECT_ALL_CODE_COMPLIANCE_RECORDS: String =
    s"""|SELECT
        | $ALL_FIELDS
        |FROM $TABLE $alias
        |JOIN ${TASKS.TABLE} t ON t.${TASKS.CI_UID} = $alias.$TASK_UID
        |JOIN ${RELEASES.TABLE} r ON r.${RELEASES.CI_UID} = t.${TASKS.RELEASE_UID}
        |LEFT OUTER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = r.${RELEASES.FOLDER_CI_UID}""".stripMargin

  private lazy val SELECT_CODE_COMPLIANCE_RECORD_BY_ID: String =
    s"""|$SELECT_ALL_CODE_COMPLIANCE_RECORDS
        |WHERE $alias.$RECORD_ID = :recordId""".stripMargin

  private lazy val SELECT_CODE_COMPLIANCE_BY_TASK_ID: String =
    s"""|$SELECT_ALL_CODE_COMPLIANCE_RECORDS
        |WHERE t.${TASKS.TASK_ID} = :taskId""".stripMargin

  private lazy val SELECT_CODE_COMPLIANCE_RECORDS_BY_RELEASE_ID: String =
    s"""|$SELECT_ALL_CODE_COMPLIANCE_RECORDS
        |WHERE r.${RELEASES.RELEASE_ID} = :releaseId""".stripMargin

  private lazy val EXISTS_CODE_COMPLIANCE_RECORD_BY_ID: String =
    s"""SELECT COUNT($RECORD_ID) FROM $TABLE WHERE $RECORD_ID = :recordId""".stripMargin

  private lazy val INSERT_CODE_COMPLIANCE_RECORD: String =
    s"""|INSERT INTO $TABLE (
        | ${commonFieldNames(CODE_COMPLIANCE_TASK_REPORTING_RECORD_LIVE)},
        | $TASK_UID,
        | $PROJECT,
        | $PROJECT_URL,
        | $ANALYSIS_DATE,
        | $OUTCOME,
        | $COMPLIANCE_DATA
        |) VALUES (
        | ${commonParameterNames()},
        | :taskUid,
        | :project,
        | :projectUrl,
        | :analysisDate,
        | :outcome,
        | :complianceData
        |)""".stripMargin

  private lazy val DELETE_CODE_COMPLIANCE_RECORD: String =
    s"""|DELETE FROM $TABLE
        |WHERE $RECORD_ID = :recordId""".stripMargin


}

class CodeComplianceRecordLiveRepository(taskPersistence: TaskPersistence,
                                         val ciIdService: CiIdService,
                                         val jdbcTemplate: JdbcTemplate,
                                         val dialect: Dialect) extends SpecializedFacetRepository(Type.valueOf(classOf[CodeComplianceRecord]))
  with FacetRepository.ForLive
  with PersistenceSupport
  with SearchRecordsSupport[CodeComplianceRecord]
  with RepositoryExceptionUtils
  with Logging {

  override def get(recordId: String, recordType: Option[Type]): Facet = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(SELECT_CODE_COMPLIANCE_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), codeComplianceRowMapper)
      .headOption.getOrElse(throw new NotFoundException(s"Record '$recordId' not found."))
  }

  override protected def doCreate(facet: Facet): Facet = {
    logger.debug("Creating CodeCompliance record in live")
    requiresInstanceOfSupportedType(facet.getType)
    withTargetTaskUid(facet.getTargetId) { taskUid =>
      val codeComplianceFacetFacet = facet.asInstanceOf[CodeComplianceRecord]
      namedTemplate.update(INSERT_CODE_COMPLIANCE_RECORD, asQueryParameters(codeComplianceFacetFacet, taskUid))
      facet.setTargetId(facet.getTargetId)
      facet
    }.getOrElse {
      throw targetTaskNotFoundException(facet.getId, facet.getTargetId)
    }
  }

  override def delete(recordId: String, recordType: Option[Type]): Unit =
    sqlUpdate(DELETE_CODE_COMPLIANCE_RECORD, params("recordId" -> getName(recordId.normalized)), _ => ())

  override def update(facet: Facet): Facet = throw new UnsupportedOperationException("You cannot update this record!")

  override def search(facetsFilters: FacetFilters): Seq[Facet] = searchRecords(facetsFilters, SELECT_ALL_CODE_COMPLIANCE_RECORDS, codeComplianceRowMapper)

  override def exists(recordId: String, recordType: Option[Type]): Boolean = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(EXISTS_CODE_COMPLIANCE_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), _.getInt(1) > 0).headOption.exists(identity)
  }

  override def findAllFacetsByRelease(release: Release): Seq[Facet] = findAllFacetsByReleaseId(release.getId).toSeq

  override def findAllFacetsByTask(task: Task): Seq[Facet] = findAllFacetsByTargetId(shortenId(task.getId)).toSeq

  override protected def findAllFacetsByTargetId(targetId: CiId): mutable.Seq[CodeComplianceRecord] =
    sqlQuery(SELECT_CODE_COMPLIANCE_BY_TASK_ID, params("taskId" -> getFolderlessId(targetId)), codeComplianceRowMapper)

  override protected def findAllFacetsByReleaseId(releaseId: CiId): mutable.Seq[CodeComplianceRecord] =
    sqlQuery(SELECT_CODE_COMPLIANCE_RECORDS_BY_RELEASE_ID, params("releaseId" -> getName(releaseId.normalized)), codeComplianceRowMapper)

  private def withTargetTaskUid[A](targetId: CiId)(f: TaskCiUid => A): Option[A] =
    taskPersistence.getTaskUidById(targetId).map(f)

  private def codeComplianceRowMapper: RowMapper[CodeComplianceRecord] = (rs: ResultSet, _: Int) => {
    val codeComplianceRecord: CodeComplianceRecord = Type.valueOf(rs.getString(CODE_COMPLIANCE_TASK_REPORTING_RECORD_LIVE.RECORD_TYPE))
      .getDescriptor.newInstance[CodeComplianceRecord](rs.getString(CODE_COMPLIANCE_TASK_REPORTING_RECORD_LIVE.RECORD_ID))
    val folderPath = Option(rs.getString(FOLDERS.FOLDER_PATH)).map(FolderId.apply).getOrElse(FolderId.Root)
    val releaseId = rs.getString(RELEASES.RELEASE_ID)
    val taskId = Ids.getReleaselessChildId(rs.getString(TASKS.TASK_ID))
    val targetId = folderPath / rs.getString(FOLDERS.FOLDER_ID) / releaseId / taskId
    codeComplianceItemRowMapper(rs, codeComplianceRecord, targetId.absolute, CODE_COMPLIANCE_TASK_REPORTING_RECORD_LIVE)
  }
}
