package com.xebialabs.xlrelease.reports.repository.deployment

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.udm.reporting.{DeploymentRecord, DeploymentStatus}
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.reports.repository.CommonRowAndParamsMapper._
import com.xebialabs.xlrelease.reports.repository.deployment.DeploymentRecordLiveRepository._
import com.xebialabs.xlrelease.reports.repository.{RepositoryExceptionUtils, SearchRecordsSupport}
import com.xebialabs.xlrelease.repository.FacetRepository.SpecializedFacetRepository
import com.xebialabs.xlrelease.repository.Ids.{getFolderlessId, getName}
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.{CiId, _}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, RELEASES, TASKS}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, TaskCiUid, TaskPersistence}
import com.xebialabs.xlrelease.repository.{FacetRepository, Ids}
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.utils.FolderId
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.collection.mutable
import scala.util.Try

object DeploymentRecordLiveRepository {

  import DEPLOYMENT_TASK_REPORTING_RECORD_LIVE._

  private lazy val ALL_FIELDS =
    s"""| ${commonFieldNames(DEPLOYMENT_TASK_REPORTING_RECORD_LIVE, "i")},
        | f.${FOLDERS.FOLDER_PATH},
        | f.${FOLDERS.FOLDER_ID},
        | r.${RELEASES.RELEASE_ID},
        | t.${TASKS.TASK_ID},
        | i.$DEPLOYMENT_TASK,
        | i.$DEPLOYMENT_TASK_URL,
        | i.$ENVIRONMENT_NAME,
        | i.$APPLICATION_NAME,
        | i.$VERSION,
        | i.$STATUS""".stripMargin

  private lazy val SELECT_ALL_DEPLOYMENT_RECORDS: String =
    s"""|SELECT
        | $ALL_FIELDS
        |FROM ${TABLE} i
        |JOIN ${TASKS.TABLE} t ON t.${TASKS.CI_UID} = i.$TASK_UID
        |JOIN ${RELEASES.TABLE} r ON r.${RELEASES.CI_UID} = t.${TASKS.RELEASE_UID}
        |LEFT OUTER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = r.${RELEASES.FOLDER_CI_UID}""".stripMargin


  private lazy val SELECT_DEPLOYMENT_RECORD_BY_ID: String =
    s"""|$SELECT_ALL_DEPLOYMENT_RECORDS
        |WHERE i.$RECORD_ID = :recordId""".stripMargin

  private lazy val SELECT_DEPLOYMENT_RECORDS_BY_TASK_ID: String =
    s"""|$SELECT_ALL_DEPLOYMENT_RECORDS
        |WHERE t.${TASKS.TASK_ID} = :taskId""".stripMargin

  private lazy val SELECT_DEPLOYMENT_RECORDS_BY_RELEASE_ID: String =
    s"""|$SELECT_ALL_DEPLOYMENT_RECORDS
        |WHERE r.${RELEASES.RELEASE_ID} = :releaseId""".stripMargin

  private lazy val EXISTS_DEPLOYMENT_RECORD_BY_ID: String =
    s"""SELECT COUNT($RECORD_ID) FROM ${TABLE} WHERE $RECORD_ID = :recordId""".stripMargin

  private lazy val EXISTS_DEPLOYMENT_RECORD_CREATED_VIA_API_BY_TASK_ID_AND_RETRY_ATTEMPT_NUMBER: String =
    s"""| SELECT COUNT($RECORD_ID) FROM $TABLE i
        |WHERE i.$TASK_UID = :taskUid
        |AND i.$RETRY_ATTEMPT_NUMBER = :retryAttemptNumber
        |AND i.$CREATED_VIA_API = 1
        |""".stripMargin

  private lazy val INSERT_DEPLOYMENT_RECORD: String =
    s"""| INSERT INTO ${TABLE} (
        | ${commonFieldNames(DEPLOYMENT_TASK_REPORTING_RECORD_LIVE)},
        | $TASK_UID,
        | $DEPLOYMENT_TASK,
        | $DEPLOYMENT_TASK_URL,
        | $ENVIRONMENT_NAME,
        | $APPLICATION_NAME,
        | $VERSION,
        | $STATUS
        |) VALUES (
        | ${commonParameterNames()},
        | :taskUid,
        | :deploymentTask,
        | :deploymentTaskUrl,
        | :environmentName,
        | :applicationName,
        | :version,
        | :status
        |)""".stripMargin

  private lazy val DELETE_DEPLOYMENT_RECORD: String =
    s"""|DELETE FROM ${TABLE}
        |WHERE $RECORD_ID = :recordId""".stripMargin

  private lazy val FIND_ALL_APPLICATION_NAMES =
    s"""SELECT DISTINCT $APPLICATION_NAME
       |FROM $TABLE""".stripMargin

  private lazy val FIND_ALL_ENVIRONMENT_NAMES =
    s"""SELECT DISTINCT $ENVIRONMENT_NAME
       |FROM $TABLE""".stripMargin

  private lazy val DELETE_AUTOGENERATED_IN_PROGRESS_FACETS: String =
    s"""|DELETE FROM ${TABLE}
        |WHERE $TASK_UID = :taskUid
        |AND $RETRY_ATTEMPT_NUMBER = :retryAttemptNumber
        |AND $STATUS = :status
        |AND $CREATED_VIA_API = :createdViaApi
        |""".stripMargin
}

class DeploymentRecordLiveRepository(taskPersistence: TaskPersistence,
                                     val ciIdService: CiIdService,
                                     val jdbcTemplate: JdbcTemplate,
                                     val dialect: Dialect) extends SpecializedFacetRepository(Type.valueOf(classOf[DeploymentRecord]))
  with FacetRepository.ForLive
  with PersistenceSupport
  with SearchRecordsSupport[DeploymentRecord]
  with RepositoryExceptionUtils
  with Logging {

  override def get(recordId: String, recordType: Option[Type]): Facet = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(SELECT_DEPLOYMENT_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), deploymentRowMapper)
      .headOption.getOrElse(throw new NotFoundException(s"Record '$recordId' not found."))
  }

  override protected def doCreate(facet: Facet): Facet = {
    logger.debug("Creating Deployment record in live")
    requiresInstanceOfSupportedType(facet.getType)
    withTargetTaskUid(facet.getTargetId) { taskUid =>
      val deploymentRecord = facet.asInstanceOf[DeploymentRecord]
      sqlInsert(INSERT_DEPLOYMENT_RECORD, asQueryParameters(deploymentRecord, taskUid))
      facet
    }.getOrElse {
      throw targetTaskNotFoundException(facet.getId, facet.getTargetId)
    }
  }

  override def delete(recordId: String, recordType: Option[Type]): Unit =
    sqlUpdate(DELETE_DEPLOYMENT_RECORD, params("recordId" -> getName(recordId.normalized)), _ => ())

  override def update(facet: Facet): Facet = throw new UnsupportedOperationException("You cannot update this record!")

  override def search(facetsFilters: FacetFilters): Seq[Facet] = searchRecords(facetsFilters, SELECT_ALL_DEPLOYMENT_RECORDS, deploymentRowMapper)

  override def exists(recordId: String, recordType: Option[Type]): Boolean = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(EXISTS_DEPLOYMENT_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), _.getInt(1) > 0).headOption.exists(identity)
  }

  override def findAllFacetsByRelease(release: Release): Seq[Facet] = findAllFacetsByReleaseId(release.getId).toSeq

  override def findAllFacetsByTask(task: Task): Seq[Facet] = findAllFacetsByTargetId(task.getId).toSeq

  def facetsCreatedViaApiExist(task: Task, retryAttemptNumber: Int): Boolean = {
    withTargetTaskUid(task) { taskUid =>
      sqlQuery(EXISTS_DEPLOYMENT_RECORD_CREATED_VIA_API_BY_TASK_ID_AND_RETRY_ATTEMPT_NUMBER, params(
        "taskUid" -> taskUid,
        "retryAttemptNumber" -> retryAttemptNumber
      ), _.getInt(1) > 0).headOption.exists(identity)
    }.getOrElse(false)
  }

  def deleteAutogeneratedInProgressFacets(targetId: CiId, retryAttemptNumber: Int): Unit = {
    withTargetTaskUid(targetId) { taskUid =>
      sqlExec(
        DELETE_AUTOGENERATED_IN_PROGRESS_FACETS,
        params(
          "taskUid" -> taskUid,
          "retryAttemptNumber" -> retryAttemptNumber,
          "status" -> DeploymentStatus.IN_PROGRESS.value(),
          "createdViaApi" -> 0
        ),
        ps => Try(ps.execute())
      )
    }
  }

  override protected def findAllFacetsByReleaseId(releaseId: CiId): mutable.Seq[DeploymentRecord] =
    sqlQuery(SELECT_DEPLOYMENT_RECORDS_BY_RELEASE_ID, params("releaseId" -> getName(releaseId.normalized)), deploymentRowMapper)

  override protected def findAllFacetsByTargetId(targetId: CiId): mutable.Seq[DeploymentRecord] =
    sqlQuery(SELECT_DEPLOYMENT_RECORDS_BY_TASK_ID, params("taskId" -> getFolderlessId(targetId)), deploymentRowMapper)

  def findAllApplicationNames: Set[String] =
    sqlQuery[String](FIND_ALL_APPLICATION_NAMES, params(), (rs: ResultSet, _: Int) => rs.getString(DEPLOYMENT_TASK_REPORTING_RECORD_LIVE.APPLICATION_NAME))
      .toSet

  def findAllEnvironmentNames: Set[String] =
    sqlQuery[String](FIND_ALL_ENVIRONMENT_NAMES, params(), (rs: ResultSet, _: Int) => rs.getString(DEPLOYMENT_TASK_REPORTING_RECORD_LIVE.ENVIRONMENT_NAME))
      .toSet

  private def withTargetTaskUid[A](task: Task)(f: TaskCiUid => A): Option[A] = {
    Option(task.getCiUid()).orElse(taskPersistence.getTaskUidById(task.getId())).map(f)
  }

  private def withTargetTaskUid[A](targetId: CiId)(f: TaskCiUid => A): Option[A] =
    taskPersistence.getTaskUidById(targetId).map(f)

  private val deploymentRowMapper: RowMapper[DeploymentRecord] = (rs: ResultSet, _: Int) => {
    val deploymentRecord: DeploymentRecord = Type.valueOf(rs.getString(DEPLOYMENT_TASK_REPORTING_RECORD_LIVE.RECORD_TYPE))
      .getDescriptor.newInstance[DeploymentRecord](rs.getString(DEPLOYMENT_TASK_REPORTING_RECORD_LIVE.RECORD_ID))
    val folderPath = Option(rs.getString(FOLDERS.FOLDER_PATH)).map(FolderId.apply).getOrElse(FolderId.Root)
    val releaseId = rs.getString(RELEASES.RELEASE_ID)
    val taskId = Ids.getReleaselessChildId(rs.getString(TASKS.TASK_ID))
    val targetId = folderPath / rs.getString(FOLDERS.FOLDER_ID) / releaseId / taskId
    deploymentItemRowMapper(rs, deploymentRecord, targetId.absolute, DEPLOYMENT_TASK_REPORTING_RECORD_LIVE)
  }

}
