package com.xebialabs.xlrelease.reports.upgrade

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.facet.{Facet, TaskReportingRecord}
import com.xebialabs.xlrelease.reports.repository.build.BuildRecordRepository
import com.xebialabs.xlrelease.reports.repository.codecompliance.CodeComplianceRecordRepository
import com.xebialabs.xlrelease.reports.repository.deployment.DeploymentRecordRepository
import com.xebialabs.xlrelease.reports.repository.itsm.ItsmRecordRepository
import com.xebialabs.xlrelease.reports.repository.plan.PlanRecordRepository
import com.xebialabs.xlrelease.repository.FacetRepositoryDispatcher
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.{FacetPersistence, FacetSqlBuilder, PersistenceSupport}
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.UpgradeSupport.{BatchSupport, TransactionSupport}
import grizzled.slf4j.Logging
import org.codehaus.jettison.json.JSONObject
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component
import org.springframework.transaction.support.TransactionTemplate

@Component
class XLRelease950LiveFacetUpgrade @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                                @Qualifier("xlrRepositoryTransactionTemplate") val transactionTemplate: TransactionTemplate,
                                                sqlRepositoryAdapter: SqlRepositoryAdapter,
                                                facetPersistence: FacetPersistence,
                                                facetRepositoryDispatcher: FacetRepositoryDispatcher,
                                                buildRecordRepository: BuildRecordRepository,
                                                planRecordRepository: PlanRecordRepository,
                                                itsmRecordRepository: ItsmRecordRepository,
                                                codeComplianceRecordRepository: CodeComplianceRecordRepository,
                                                deploymentRecordRepository: DeploymentRecordRepository,
                                                xlrConfig: XlrConfig)
  extends Upgrade
    with PersistenceSupport
    with Logging
    with BatchSupport
    with TransactionSupport {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "9.5.0#1")

  override def doUpgrade(): Boolean = {
    logger.info("Starting immutable facet storage upgrade for live database")
    val supportedTypes = facetRepositoryDispatcher.supportedTypes ++ deprecatedFacetTypes
    logger.debug(s"Supported types are: ${supportedTypes.map(_.toString).mkString(", ")}")

    supportedTypes.foreach { facetType =>
      logger.trace(s"migrating facets of type '${facetType.toString}'")
      val facetsInFacetsTable = new FacetSqlBuilder().select.withFacetTypes(Seq(facetType)).build()
      val facets = facetPersistence.search(facetsInFacetsTable)
      logger.trace(s"found ${facets.length} facets of type ${facetType.toString}")
      doInBatch(facets, itemsName = facetType.getName) { batch =>
        batch.items.foreach { facetRow =>
          val recordId = facetRow.facetId

          deserializeFacet[TaskReportingRecord](new JSONObject(facetRow.content)).foreach { record =>
            logger.trace(s"processing facet ${record.getId}/${record.getType.toString}")
            setDefaultValuesForRequiredProperties(record)
            renewDeprecatedTypes(record)

            doInTransaction {
              insertToStrictTable(record)
              deleteFromFacetsTable(recordId)
            }
          }
        }
      }

      logger.info(s"Upgrade for ${facetType.getName} completed")
    }
    true
  }

  private def deserializeFacet[T <: TaskReportingRecord](facetJson: JSONObject): Option[T] = {
    val facetOption = sqlRepositoryAdapter.deserialize[T](facetJson.toString)
    facetOption.foreach(f => f.setTargetId(facetJson.getString("targetId")))
    facetOption
  }

  private def insertToStrictTable(facet: Facet): Unit = {
    try {
      facetRepositoryDispatcher.liveRepository.create(facet)
    } catch {
      case e: NotFoundException => logger.warn(e.getMessage)
    }
  }

  private def deleteFromFacetsTable(facetId: String): Unit = {
    facetPersistence.delete(facetId)
  }

}
