package com.xebialabs.xlrelease.reports.job.api

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.reports.job.api.ReportingEngineService.ReportJobId
import com.xebialabs.xlrelease.storage.Storage
import com.xebialabs.xlrelease.utils.CloseableUtils.using
import grizzled.slf4j.Logging
import org.springframework.util.StreamUtils.copy

import java.io.{FileNotFoundException, InputStream, OutputStream}
import java.net.URI
import java.util.Date

trait ReportResult {
  def resultUri: String

  def generatedOn: Date

  def reportName: String

  def contentType: String

  def jobId: ReportJobId

  def fileName: String = {
    val uri = new URI(resultUri)
    val path = Storage.toPath(uri)
    path.getFileName.toString
  }
}

trait StreamingReportResult extends ReportResult {
  def write(outputStream: OutputStream): Unit
}

abstract class AbstractStreamingReportResult(reportResult: ReportResult)
  extends StreamingReportResult with Logging {
  override def write(outputStream: OutputStream): Unit = {
    try {
      val stream = inputStream()
      using(stream) { inputStream =>
        copy(inputStream, outputStream)
      }
    } catch {
      case _: FileNotFoundException =>
        throw new NotFoundException(s"'%s' was not found", reportResult.resultUri)
      case ex: Exception =>
        logger.error(s"Unable to access '${reportResult.resultUri}'", ex)
    }
  }

  override def resultUri: String = resolvedUri.toString

  override def generatedOn: Date = reportResult.generatedOn

  override def reportName: String = reportResult.reportName

  override def contentType: String = reportResult.contentType

  override def jobId: ReportJobId = reportResult.jobId

  protected val resolvedUri: URI

  protected def inputStream(): InputStream
}
