package com.xebialabs.xlrelease.reports.repository.plan

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters
import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.udm.reporting.PlanRecord
import com.xebialabs.xlrelease.domain.utils.FullReleaseId
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.reports.repository.CommonRowAndParamsMapper._
import com.xebialabs.xlrelease.reports.repository.plan.PlanRecordArchiveRepository._
import com.xebialabs.xlrelease.reports.repository.{ArchivedTasksRepository, RepositoryExceptionUtils, SearchRecordsSupport}
import com.xebialabs.xlrelease.repository.FacetRepository
import com.xebialabs.xlrelease.repository.FacetRepository.SpecializedFacetRepository
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.collection.mutable
import scala.util.Failure

object PlanRecordArchiveRepository {

  import PLAN_TASK_REPORTING_RECORD_ARCHIVE._

  private lazy val ALL_FIELDS =
    s"""| ${commonFieldNames(PLAN_TASK_REPORTING_RECORD_ARCHIVE, "p")},
        | t.$REPORT_TASKS_ID_COLUMN,
        | p.$TICKET,
        | p.$TITLE,
        | p.$TICKET_TYPE,
        | p.$STATUS,
        | p.$TICKET_URL,
        | p.$UPDATED_DATE,
        | p.$UPDATED_BY""".stripMargin

  private lazy val SELECT_ALL_PLAN_RECORDS: String =
    s"""|SELECT
        | $ALL_FIELDS
        |FROM $TABLE p
        |JOIN $REPORT_TASKS_TABLE_NAME t ON t.$REPORT_TASKS_UID_COLUMN = p.$TASK_UID""".stripMargin

  private lazy val SELECT_PLAN_RECORD_BY_ID_STMT: String =
    s"""|$SELECT_ALL_PLAN_RECORDS
        |WHERE p.$RECORD_ID = :recordId""".stripMargin

  private lazy val SELECT_PLAN_RECORDS_BY_TASK_ID_STMT =
    s"""|$SELECT_ALL_PLAN_RECORDS
        |WHERE t.$REPORT_TASKS_ID_COLUMN = :taskId""".stripMargin

  private lazy val SELECT_PLAN_RECORDS_BY_RELEASE_ID_STMT: String =
    s"""|$SELECT_ALL_PLAN_RECORDS
        |JOIN $REPORT_RELEASES_TABLE_NAME r ON r.$REPORT_RELEASES_ID_COLUMN = t.$REPORT_TASKS_RELEASEID_COLUMN
        |WHERE r.$REPORT_RELEASES_ID_COLUMN = :releaseId AND r.$REPORT_RELEASES_PRE_ARCHIVED <> 1""".stripMargin

  private lazy val EXISTS_PLAN_RECORD_BY_ID: String =
    s"""SELECT COUNT($RECORD_ID) FROM $TABLE WHERE $RECORD_ID = :recordId""".stripMargin

  private lazy val INSERT_PLAN_RECORD_STMT =
    s"""| INSERT INTO $TABLE (
       | ${commonFieldNames(PLAN_TASK_REPORTING_RECORD_ARCHIVE)},
       | $TASK_UID,
       | $TICKET,
       | $TITLE,
       | $TICKET_TYPE,
       | $STATUS,
       | $UPDATED_DATE,
       | $UPDATED_BY,
       | $TICKET_URL
       |) VALUES (
       | ${commonParameterNames()},
       | :taskUid,
       | :ticket,
       | :title,
       | :ticketType,
       | :status,
       | :updatedDate,
       | :updatedBy,
       | :ticketUrl
       |)""".stripMargin

  private lazy val DELETE_PLAN_RECORD_STMT: String =
    s"""|DELETE FROM $TABLE
        |WHERE $RECORD_ID = :recordId""".stripMargin

}

class PlanRecordArchiveRepository(archivedTasksRepository: ArchivedTasksRepository,
                                  val ciIdService: CiIdService,
                                  val jdbcTemplate: JdbcTemplate,
                                  val dialect: Dialect)
  extends SpecializedFacetRepository(Type.valueOf(classOf[PlanRecord]))
    with FacetRepository.ForArchive
    with PersistenceSupport
    with SearchRecordsSupport[PlanRecord]
    with RepositoryExceptionUtils
    with Logging {

  override def get(recordId: String, recordType: Option[Type]): Facet = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(SELECT_PLAN_RECORD_BY_ID_STMT, params("recordId" -> getName(recordId.normalized)), planRowMapper)
      .headOption.getOrElse(throw new NotFoundException(s"Record '$recordId' not found."))
  }

  override protected def doCreate(facet: Facet): Facet = {
    logger.debug("Creating Plan record in archive")
    requiresInstanceOfSupportedType(facet.getType)
    archivedTasksRepository.taskUidById(shortenId(facet.getTargetId)).map { taskUid =>
      val planRecord = facet.asInstanceOf[PlanRecord]
      sqlInsert(INSERT_PLAN_RECORD_STMT, asQueryParameters(planRecord, taskUid))
      facet
    }.recoverWith {
      case _: NotFoundException =>
        Failure(targetTaskNotFoundException(facet.getId, facet.getTargetId))
    }.get
  }

  override def delete(recordId: String, facetType: Option[Type]): Unit =
    sqlUpdate(DELETE_PLAN_RECORD_STMT, params("recordId" -> getName(recordId.normalized)), _ => ())

  override def update(facet: Facet): Facet = throw new UnsupportedOperationException("You cannot update records!")

  override def search(facetsFilters: FacetFilters): Seq[Facet] = searchRecords(facetsFilters, SELECT_ALL_PLAN_RECORDS, planRowMapper)

  override def exists(recordId: String, recordType: Option[Type]): Boolean = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(EXISTS_PLAN_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), _.getInt(1) > 0).headOption.exists(identity)
  }

  override def findAllFacetsByRelease(release: Release): Seq[Facet] = findAllFacetsByReleaseId(release.getId).toSeq

  override def findAllFacetsByTask(task: Task): Seq[Facet] = findAllFacetsByTargetId(shortenId(task.getId)).toSeq

  override protected def findAllFacetsByTargetId(targetId: CiId): mutable.Seq[PlanRecord] =
    sqlQuery(SELECT_PLAN_RECORDS_BY_TASK_ID_STMT, params("taskId" -> targetId), planRowMapper)

  override protected def findAllFacetsByReleaseId(releaseId: CiId): mutable.Seq[PlanRecord] =
    sqlQuery[PlanRecord](SELECT_PLAN_RECORDS_BY_RELEASE_ID_STMT, params(
      "releaseId" -> FullReleaseId(releaseId).withOnlyOneParentOrApplicationsForArchiveDb()
    ), planRowMapper)

  private val planRowMapper: RowMapper[PlanRecord] = (rs: ResultSet, _: Int) => {
    val planRecord: PlanRecord = Type.valueOf(rs.getString(PLAN_TASK_REPORTING_RECORD_ARCHIVE.RECORD_TYPE))
      .getDescriptor.newInstance[PlanRecord](rs.getString(PLAN_TASK_REPORTING_RECORD_ARCHIVE.RECORD_ID))
    planItemRowMapper(rs, planRecord, rs.getString(REPORT_TASKS_ID_COLUMN), PLAN_TASK_REPORTING_RECORD_ARCHIVE)
  }
}


