package com.xebialabs.xlrelease.reports.excel

import com.xebialabs.xlrelease.domain.ReleaseKind
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.reports.audit.{CommonFormat, InstanceData}
import com.xebialabs.xlrelease.reports.domain.MaybeData._
import com.xebialabs.xlrelease.reports.domain.{MaybeData, ReleaseStartedBy, ReleaseTitleAndPath}
import com.xebialabs.xlrelease.reports.excel.MultiReleaseAuditReport._
import org.apache.poi.common.usermodel.HyperlinkType
import org.apache.poi.ss.usermodel.Workbook
import org.apache.poi.xssf.usermodel.XSSFCellStyle

import java.io.File
import java.util.Date
import scala.util.{Failure, Success}

object MultiReleaseAuditReport {
  val RELEASE_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("Title", 26)
  val RELEASE_FOLDER_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("Folder", 35)
  val RELEASE_TYPE_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("Type", CommonFormat.DATE_WIDTH)
  val STATUS_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("Status", 14) // to fit the "from" field in release filters
  val RELEASE_OWNER_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("Owner", 25)
  val START_DATE_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("Start date", CommonFormat.DATE_WIDTH)
  val END_DATE_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("End date", CommonFormat.DATE_WIDTH)
  val DURATION_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("Duration", 16)
  val STARTED_BY_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("Started by", 14)
  val FROM_TEMPLATE_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("Template", 26)
  val FROM_TEMPLATE_FOLDER_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("Template folder", 22)

  val RELEASES_TABLE_COLUMNS = 8

  val LOGO_WIDTH: Int = 53 // found by experiment, magic units
  val LOGO_HEIGHT: Int = 9 // found by experiment, magic units
  val CELLS_IN_ROW: Int = 11

  def getWorkBook(data: Data): Workbook =
    new MultiReleaseAuditReport(data).getWorkBook

  case class PerReleaseData(title: String,
                            owner: String,
                            folder: Maybe[String],
                            startDate: Date,
                            endOrDueDate: Either[Date, Date],
                            duration: String,
                            kind: ReleaseKind,
                            status: ReleaseStatus,
                            startedBy: Option[ReleaseStartedBy],
                            storedPath: Option[(File, String)],
                            templateInfo: Maybe[Option[ReleaseTitleAndPath]])

  case class Data(generatedBy: String,
                  generatedOn: Date,
                  instanceData: InstanceData,
                  searchCriteria: Seq[FilterData],
                  releases: Seq[Maybe[PerReleaseData]])

  case class FilterData(label: String, values: Seq[Any])


}

class MultiReleaseAuditReport(data: MultiReleaseAuditReport.Data)
  extends ExcelCommonHeader with ErrorHandler {

  override val workbook: ReportWorkbook = new ReportWorkbook
  override val styles: ExcelStyles = workbook.getStyles
  var sheetWriter: ExcelSheetWriter = workbook.createReportSection("Releases Overview")

  def getWorkBook: Workbook = {
    addCommonHeader(data.generatedBy, data.generatedOn, data.instanceData,
      CELLS_IN_ROW, LOGO_WIDTH, LOGO_HEIGHT, ReportInfo())
    addSearchCriteriaFilterHeader(data.searchCriteria)
    fillCellsUntilEndOfRow(CELLS_IN_ROW, styles.noGrid)
    addContent()
    workbook.makeWorkbookWithErrors(styles)
  }

  private def addContent(): Unit = {
    addHeader()
    sheetWriter.addHeaderFilter(RELEASES_TABLE_COLUMNS)

    data.releases
      .sortBy(_.map(_.getValue.startDate).getOrElse(new Date(0)))
      .foreach(addReleaseRow)

    sheetWriter.newRow()
  }

  def addHeader(): Unit = {
    sheetWriter.newRow()
    sheetWriter.addHeaderCell(RELEASE_COLUMN, styles.whiteOnGreenBig)
    sheetWriter.addHeaderCell(RELEASE_FOLDER_COLUMN, styles.whiteOnGreenBig)
    sheetWriter.addHeaderCell(RELEASE_TYPE_COLUMN, styles.whiteOnGreenBig)
    sheetWriter.addHeaderCell(STATUS_COLUMN, styles.whiteOnGreenBig)
    sheetWriter.addHeaderCell(RELEASE_OWNER_COLUMN, styles.whiteOnGreenBig)
    sheetWriter.addHeaderCell(START_DATE_COLUMN, styles.whiteOnGreenBig)
    sheetWriter.addHeaderCell(END_DATE_COLUMN, styles.whiteOnGreenBig)
    sheetWriter.addHeaderCell(DURATION_COLUMN, styles.whiteOnGreenBig)
    sheetWriter.addHeaderCell(STARTED_BY_COLUMN, styles.whiteOnGreenBig)
    sheetWriter.addHeaderCell(FROM_TEMPLATE_COLUMN, styles.whiteOnGreenBig)
    sheetWriter.addHeaderCell(FROM_TEMPLATE_FOLDER_COLUMN, styles.whiteOnGreenBig)
  }

  private def addErrorRow(throwable: Throwable): Unit = {
    addCellForMaybe(MaybeData.failure(throwable));
  }

  private def addReleaseRow(release: Maybe[PerReleaseData]): Unit = {
    sheetWriter.newRow()
      .setRowHeight(17)
    release match {
      case Failure(t) => addErrorRow(t)
      case Success(s) =>
        val release = s.getValue
        val endDate: Date = release.endOrDueDate.fold[Date](d => d, d => d)
        val fileAddress = release.storedPath.getOrElse((None, "#"))._2.replace(" ", "%20").replace("\\", "/")

        s match {
          case Left((t, _)) =>
            addCellForMaybe(MaybeData.partial(t, release.title))
          case Right(_) =>
            sheetWriter.addHyperlink(release.title, fileAddress, HyperlinkType.FILE, styles.hyperlink)
        }
        addCellForMaybe(release.folder, styles.default)
        addCellForMaybe(MaybeData.nonEmptyString(CommonFormat.prettifyStatusString(release.kind.value())), styles.default)
        addCellForMaybe(MaybeData.nonEmptyString(CommonFormat.prettifyStatusString(release.status.value())), styles.default)
        addCellForMaybe(MaybeData.allowingNull(release.owner), styles.default)
        addCellForMaybe(MaybeData.allowingNull(release.startDate), styles.date)
        addCellForMaybe(MaybeData.allowingNull(endDate), styles.date)
        addCellForMaybe(MaybeData.allowingNull(release.duration))
        addCellForMaybe(CommonFormat.formatStartedBy(release.startedBy), styles.default)
        addCellForMaybe(CommonFormat.getMaybeReleaseTitle(release.templateInfo), styles.default)
        addCellForMaybe(CommonFormat.getMaybeReleasePath(release.templateInfo), styles.default)
    }
  }

  private def addSearchCriteriaFilterHeader(filters: Seq[FilterData]): Unit = {
    // move pointer in same row as XLR version
    sheetWriter.goToStartOfLine()

    val (workflows, releases) = data.releases.partition(_.map(_.getValue.kind == ReleaseKind.WORKFLOW).getOrElse(false))

    addRow("Number of releases in report", releases.size)
    addRow("Number of workflow executions in report", workflows.size)
    addRow("", "")
    sheetWriter
      .addCell("Release filters", styles.noGridBoldText)
      .goToStartOfLine()

    filters.foreach {
      case FilterData(label, values) =>
        addRow(label, values.head)
        values.tail.foreach(addRow("", _))
    }

  }

  private def addHeaderRow(sheetWriter: ExcelSheetWriter, label: String, value: Any): Unit = {
    sheetWriter
      .addEmptyCell(styles.noGrid)
      .addCell(label, styles.leftAlignedNoGrid)
    value match {
      case (value: String, style: XSSFCellStyle) => sheetWriter.addCell(value, style)
      case value: String => sheetWriter.addCell(value, styles.noGrid)
      case value: Int => sheetWriter.addCell(value.asInstanceOf[java.lang.Integer], styles.noGridNumeric)
      case date: Date =>
        styles.noGrid.setDataFormat(sheetWriter.getDateStyle.getDataFormat) // TODO fix this
        sheetWriter.addCell(date, styles.noGrid)
      case _ => sheetWriter.addCell(value)
    }
    fillCellsUntilEndOfRow(CELLS_IN_ROW, styles.noGrid)
    sheetWriter.newRow()
  }

  private def addRow(label: String, value: Any): Unit = {
    addHeaderRow(sheetWriter, label, value)
  }

}
