package com.xebialabs.xlrelease.reports.repository

import com.google.common.io.CharStreams
import com.xebialabs.xlrelease.domain.facet.TaskReportingRecord
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.TaskRecordTable
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.{FacetPersistence, TaskCiUid}
import org.springframework.jdbc.core.support.SqlLobValue

import java.io.{ByteArrayInputStream, InputStreamReader}
import java.nio.charset.StandardCharsets.UTF_8
import java.sql.ResultSet
import java.util.Date

object CommonRowAndParamsMapper {

  val SHORT_VARCHAR = 255
  val LONG_VARCHAR = 1024
  val URL_VARCHAR = 2084

  def commonFieldNames(tableTemplate: TaskRecordTable, tableAlias: String): String = {
    val prefix = if (tableAlias.isEmpty) "" else s"$tableAlias."
    s""" $prefix${tableTemplate.RECORD_ID},
       | $prefix${tableTemplate.RECORD_TYPE},
       | $prefix${tableTemplate.SERVER_URL},
       | $prefix${tableTemplate.SERVER_USER},
       | $prefix${tableTemplate.RETRY_ATTEMPT_NUMBER},
       | $prefix${tableTemplate.CREATED_VIA_API},
       | $prefix${tableTemplate.CREATION_DATE}""".stripMargin
  }

  def commonFieldNames(tableTemplate: TaskRecordTable): String = {
    commonFieldNames(tableTemplate, "")
  }

  def commonParameterNames(): String = {
    s""":recordId,
       | :recordType,
       | :serverUrl,
       | :serverUser,
       | :retryAttemptNumber,
       | :createdViaApi,
       | :creationDate""".stripMargin
  }

  def mapCommonFields(rs: ResultSet, facet: TaskReportingRecord, targetId: String, table: TaskRecordTable): Unit = {
    facet.setTargetId(targetId)
    facet.setServerUrl(rs.getString(table.SERVER_URL))
    facet.setServerUser(rs.getString(table.SERVER_USER))
    facet.setRetryAttemptNumber(rs.getInt(table.RETRY_ATTEMPT_NUMBER))
    facet.setCreatedViaApi(rs.getInt(table.CREATED_VIA_API).asBoolean)
    facet.setCreationDate(Date.from(rs.getTimestamp(table.CREATION_DATE).toInstant))
  }

  def mapCommonParams(facet: TaskReportingRecord, taskUid: TaskCiUid): Map[String, Any] = {
    params(
      "taskUid" -> taskUid,
      "recordId" -> getName(facet.getId.normalized).truncate(SHORT_VARCHAR),
      "recordType" -> facet.getType.toString.truncate(SHORT_VARCHAR),
      "serverUrl" -> facet.getServerUrl.truncate(URL_VARCHAR),
      "serverUser" -> facet.getServerUser.truncate(SHORT_VARCHAR),
      "retryAttemptNumber" -> FacetPersistence.extractRetryAttemptNumber(facet),
      "createdViaApi" -> FacetPersistence.extractCreatedViaApi(facet),
      "creationDate" -> facet.getCreationDate
    )
  }

  def readBlobContent(rs: ResultSet, columnName: String): String = {
    val inputStream = rs.getBinaryStream(columnName)
    try {
      CharStreams.toString(new InputStreamReader(inputStream, UTF_8))
    } finally {
      inputStream.close()
    }
  }

  def createSqlLob(value: String): SqlLobValue = {
    val bytes = Option(value).map(_.getBytes(UTF_8)).getOrElse(Array.empty[Byte])
    new SqlLobValue(new ByteArrayInputStream(bytes), bytes.size)
  }
}
