package com.xebialabs.xlrelease.reports.audit

import com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName
import com.xebialabs.xlrelease.activity.ReleaseActivityDateFormatter.formatDate
import com.xebialabs.xlrelease.reports.audit.ReleasePermissionReport._
import com.xebialabs.xlrelease.reports.domain.exceptions.NoPermissionException
import com.xebialabs.xlrelease.reports.domain.{MaybeData, PermissionLabels, ReportSheetError}
import com.xebialabs.xlrelease.reports.excel._
import com.xebialabs.xlrelease.security.sql.snapshots.domain.{ContainerTeamsAndRolesForPrincipalAndPermission, RolesSnapshotUnavailableError}
import org.apache.poi.ss.util.CellRangeAddress

object ReleasePermissionReport {
  val PERMISSIONS_LABEL = "Permissions"
  val PRINCIPAL_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("Principal", 30)
  val FULL_NAME_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("Full name", 30)
  val HAS_PERMISSION_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("Has permission", 30)
  val ROLE_COLUMN: ExcelHeaderColumn = ExcelHeaderColumn("Because of team (global role)", 40)
  val HEADER_WARNING = "This report captures all permissions granted to users during the execution of the release, even if the permission was later revoked."

  case class Data(maybeContainers: Option[Seq[ContainerTeamsAndRolesForPrincipalAndPermission]], snapshotErrors: Option[Seq[RolesSnapshotUnavailableError]])

  def addSection(permissionData: Data, workbook: ReportWorkbook, styles: ExcelStyles): Unit =
    new ReleasePermissionReport(permissionData, workbook, styles).addContent()
}

case class ReleasePermissionReport(data: ReleasePermissionReport.Data, workbook: ReportWorkbook, styles: ExcelStyles) extends ErrorHandler {
  var sheetWriter: ExcelSheetWriter = _

  def addContent(): Unit = {
    if (data.maybeContainers.isEmpty) {
      workbook.addSheetError(ReportSheetError(PERMISSIONS_LABEL,
        NoPermissionException(s"User $getAuthenticatedUserName doesn't have access to view permission tab in the report")
      ))
    } else {
      sheetWriter = workbook.createReportSection(PERMISSIONS_LABEL)
      addErrorRows()
      addHeaderRow()
      data.maybeContainers.get.foreach(addPermissionRow)
    }
  }

  def addErrorRows(): Unit = {
    data.snapshotErrors.getOrElse(List()).foreach(row => {
      sheetWriter.newRow()
      val recovered = row.usedDate.map(date => s"Using the next permission record from [${formatDate(date)}].").getOrElse("Failed to create new record, see logs for details.")
      val message = s"Permission records not found for [${row.containerId.getOrElse("/")}] on [${formatDate(row.date)}]. ${recovered}"
      addCellForMaybe(MaybeData.partial(new IllegalStateException(message), message))
    })
    if (data.snapshotErrors.nonEmpty) {
      sheetWriter.newRow()
    }
  }

  def addHeaderWarning(): Unit = {
    sheetWriter.newRow()
    sheetWriter.mergeCells(new CellRangeAddress(1, 1, 0, 3))
    sheetWriter.addCell(HEADER_WARNING, styles.addBackgroundColor(styles.default, ExcelStyles.YELLOW))
    sheetWriter.newRow()
    sheetWriter.newRow()
  }

  def addHeaderRow(): Unit = {
    // addHeaderWarning() - we will not show the warning to not confuse customers with permission snapshots disabled
    Seq(
      PRINCIPAL_COLUMN,
      FULL_NAME_COLUMN,
      HAS_PERMISSION_COLUMN,
      ROLE_COLUMN
    ).foreach(sheetWriter.addHeaderCell(_, styles.whiteOnGreenBig))
  }

  def addPermissionRow(permission: ContainerTeamsAndRolesForPrincipalAndPermission): Unit = {
    sheetWriter.newRow()
    addCellForMaybe(MaybeData(permission.principal))
    addCellForMaybe(MaybeData(permission.fullName))
    addCellForMaybe(MaybeData(PermissionLabels.getLabel(permission.permission)))
    addCellForMaybe(MaybeData(permission.links.mkString(", ")))
  }
}
