package com.xebialabs.xlrelease.reports.audit

import com.xebialabs.deployit.plugin.api.reflect.PropertyKind.DATE
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind, Type}
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.facet.TaskReportingRecord
import com.xebialabs.xlrelease.domain.udm.reporting.ItsmRecord
import com.xebialabs.xlrelease.planner.PlannerReleaseItem
import com.xebialabs.xlrelease.reports.audit.CommonFormat.TaskAgent
import com.xebialabs.xlrelease.reports.audit.TaskRecordReport.{SERVER_URL_PROPERTY, TaskRows, extraHiddenProperties}
import com.xebialabs.xlrelease.reports.excel.{ErrorHandler, ExcelSheetWriter, ExcelStyles, ReportWorkbook}

import java.net.{URI, URISyntaxException}
import java.util.Date
import scala.jdk.CollectionConverters._

object TaskRecordReport {
  protected val taskReportingRecordType: Type = Type.valueOf(classOf[TaskReportingRecord])

  val SERVER_URL_PROPERTY = "serverUrl"

  // properties to hide. if the Set of types is empty it applies to all record types, otherwise only to types and sub-types not in the set.
  val extraHiddenProperties: Map[String, Set[Type]] = Map(
    "targetId" -> Set.empty,
    "variableMapping" -> Set.empty,
    "ticket_url" -> Set.empty,
    "build_url" -> Set.empty,
    "project_url" -> Set.empty,
    "record_url" -> Set.empty,
    "deploymentTask_url" -> Set.empty,
    "creationDate" -> Set(Type.valueOf(classOf[ItsmRecord]))
  )

  case class TaskRows(plannedTask: PlannerReleaseItem, taskAgent: TaskAgent, records: Seq[TaskReportingRecord])

  case class Data(recordType: Type,
                  release: Release,
                  tasks: Seq[TaskRows])

  def addSection(data: Data, workbook: ReportWorkbook, styles: ExcelStyles): Unit = {
    new TaskRecordReport(data, workbook, styles).addContent()
  }
}

class TaskRecordReport(data: TaskRecordReport.Data,
                       val workbook: ReportWorkbook,
                       val styles: ExcelStyles) extends CommonTaskReportSheet with ErrorHandler {

  lazy val sectionLabel: String = Option(data.recordType.getDescriptor.getLabel).getOrElse(data.recordType.getName.capitalize)

  var sheetWriter: ExcelSheetWriter = _

  lazy val propertyDescriptors: Seq[PropertyDescriptor] =
    data.recordType.getDescriptor.getPropertyDescriptors.asScala.toSeq
      .filterNot(pd =>
        pd.isHidden || (
          extraHiddenProperties.contains(pd.getName) && !extraHiddenProperties(pd.getName).exists(data.recordType.instanceOf)
        )
      )

  def addContent(): Unit = {
    if (data.tasks.nonEmpty) {
      sheetWriter = workbook.createReportSection(sectionLabel)
      addHeaderRow()
      data.tasks.foreach(addTaskRows)
    }
  }

  def addHeaderRow(): Unit = {
    sheetWriter.newRow()
    addTaskCommonHeader()
    propertyDescriptors.foreach { pd =>
      val header = Option(pd.getLabel).getOrElse(pd.getName).toLowerCase.capitalize
      pd.getKind match {
        case DATE => sheetWriter.addHeaderCell(header, CommonFormat.DATE_WIDTH, styles.whiteOnGreen)
        case _ => sheetWriter.addHeaderCell(header, 20, styles.whiteOnGreen)
      }
    }
    sheetWriter.newRow()
  }

  def addRecordCells(record: TaskReportingRecord): Unit = {
    def checkIfUrlPropertyExists(pd: PropertyDescriptor): Boolean = record.hasProperty(s"${pd.getName}_url")

    def propertyIsServerUrlAndIsValid(pd: PropertyDescriptor): Boolean = pd.getName == SERVER_URL_PROPERTY && isValidUrl(record.getProperty[String](pd.getName))

    def decorateWithHyperlink(pd: PropertyDescriptor): Unit = {
      val hyperlinkValue = Option(record.getProperty[Any](pd.getName)).map(_.toString).orNull
      Option(record.getProperty[Any](s"${pd.getName}_url")).map(_.toString).filter(isValidUrl) match {
        case Some(url) =>
          sheetWriter.addHyperlink(hyperlinkValue, url, styles.hyperlink)
        case None =>
          sheetWriter.addCell(hyperlinkValue, styles.default)
      }
    }

    propertyDescriptors.foreach { pd =>
      pd.getKind match {
        case PropertyKind.DATE =>
          sheetWriter.addCell(record.getProperty[Date](pd.getName), styles.date)
        case PropertyKind.INTEGER =>
          sheetWriter.addCell(record.getProperty[Integer](pd.getName), styles.default)
        case PropertyKind.STRING if propertyIsServerUrlAndIsValid(pd) =>
          sheetWriter.addHyperlink(record.getProperty[String](pd.getName), record.getProperty[String](pd.getName), styles.hyperlink)
        case _ =>
          if (checkIfUrlPropertyExists(pd)) {
            decorateWithHyperlink(pd)
          } else {
            sheetWriter.addCell(Option(record.getProperty[Any](pd.getName)).map(_.toString).orNull, styles.default)
          }
      }
    }
  }

  def addTaskRows(taskRow: TaskRows): Unit = {
    taskRow.records.foreach { record =>
      addTaskCommonCells(taskRow.plannedTask, data.release, taskRow.taskAgent)
      addRecordCells(record)
      sheetWriter.newRow()
    }
  }

  private def isValidUrl(url: String): Boolean = {
    var isValid = false
    if (url != null) {
      try {
        new URI(url)
        isValid = true
      } catch {
        case x: URISyntaxException => {
        }
      }
    }
    isValid
  }

}
