package com.xebialabs.xlrelease.reports.job.handlers

import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.configuration.AuditReportSettings
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.notifications.{AuditReportJobCompleted, NotificationService, ReportJobAborted, ReportJobFailed}
import com.xebialabs.xlrelease.reports.job.api.ReportDefinition
import com.xebialabs.xlrelease.reports.job.domain.BaseReportDefinition
import com.xebialabs.xlrelease.reports.job.events.{ReportJobAbortedEvent, ReportJobCompletedEvent, ReportJobFailedEvent}
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.io.File
import java.net.URI
import scala.util.{Failure, Success, Try}

@Component
@EventListener
class ReportJobNotificationEventHandler @Autowired()(notificationService: NotificationService,
                                                     serverConfiguration: ServerConfiguration,
                                                     configurationRepository: ConfigurationRepository
                                                    ) extends Logging {
  private val ReportDefinitionType = Type.valueOf(classOf[ReportDefinition])

  @AsyncSubscribe
  def onReportJobCompleted(event: ReportJobCompletedEvent): Unit = {
    val reportJob = event.reportJob
    val fileName = getReportFileName(reportJob.resultUri).getOrElse(reportJob.reportName)
    val AuditReport = Type.valueOf(classOf[BaseReportDefinition])
    val reportSettings = configurationRepository.read(AuditReportSettings.AUDIT_REPORT_SETTINGS_ID).asInstanceOf[AuditReportSettings]

    Type.valueOf(reportJob.reportType) match {
      case t if AuditReport.isSuperTypeOf(t) =>
        val retentionPeriod = reportSettings.getReportsRetentionPeriod
        notificationService.notify(AuditReportJobCompleted(
          s"${serverConfiguration.getServerUrl}#/audit-report-download/${reportJob.getJobId()}", fileName, retentionPeriod, reportJob.username))
      case _ =>
        logger.warn(s"Unable to send job completion notification for report type: $reportJob.reportType")
    }
  }

  @AsyncSubscribe
  def onReportJobFailed(event: ReportJobFailedEvent): Unit = {
    val reportJob = event.reportJob

    Type.valueOf(reportJob.reportType) match {
      case r if ReportDefinitionType.isSuperTypeOf(r) => notificationService.notify(ReportJobFailed(reportJob.reportName, reportJob.username))
      case _ => logger.warn(s"Unable to send job failure notification for report type: $reportJob.reportType")
    }
  }

  @AsyncSubscribe
  def onReportJobAbort(event: ReportJobAbortedEvent): Unit = {
    val reportJob = event.reportJob

    Type.valueOf(reportJob.reportType) match {
      case r if ReportDefinitionType.isSuperTypeOf(r) => notificationService.notify(ReportJobAborted(reportJob.reportName, reportJob.username))
      case _ => logger.warn(s"Unable to send job abort notification for report type: $reportJob.reportType")
    }
  }

  private def getReportFileName(resultUri: String): Option[String] = {
    Try(new URI(resultUri)) match {
      case Success(uri) =>
        Some(new File(uri.getPath).getName)
      case Failure(_) =>
        logger.warn(s"Unable to get file name from $resultUri. Setting default filename.")
        None
    }
  }
}
