package com.xebialabs.xlrelease.repository

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.domain.{Release, ReleaseExtension}
import com.xebialabs.xlrelease.repository.Ids.{getName, releaseIdFrom}
import com.xebialabs.xlrelease.service.ReleaseService

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

abstract class ReleaseExtensionsHelperRepository[T <: ReleaseExtension]
  (releaseService: ReleaseService, releaseExtensionsRepository: ReleaseExtensionsRepository) {

  @Timed
  def findByIdOrDefault(release: Release, releaseExtensionId: String): T = {
    release.getExtensions.asScala
      .find(extension => getName(extension.getId) == getName(releaseExtensionId))
      .map(_.asInstanceOf[T])
      .getOrElse(createDefaultReleaseExtension(releaseExtensionId))
  }

  @Timed
  def findByIdOrDefault(releaseExtensionId: String): T = {
    val extensionOption: Option[T] = getReleaseExtension(releaseExtensionId)
    val result: T = extensionOption.getOrElse(createDefaultReleaseExtension(releaseExtensionId))
    result
  }

  @Timed
  def findById(releaseExtensionId: String): T = {
    val extensionOption: Option[T] = getReleaseExtension(releaseExtensionId)
    val result: T = extensionOption.getOrElse {
      throw new NotFoundException(s"ReleaseExtension [%s] does not exist in the repository or archive", releaseExtensionId)
    }
    result
  }

  @Timed
  def getReleaseExtension(releaseExtensionId: String): Option[T] = {
    val releaseId = releaseIdFrom(releaseExtensionId)
    val extensionOption = if (releaseService.exists(releaseId)) {
      releaseExtensionsRepository.read(releaseExtensionId)
    } else if (releaseService.isArchived(releaseId)) {
      getExtensionFromArchivedRelease(releaseExtensionId)
    } else {
      None
    }
    extensionOption
  }

  private def getExtensionFromArchivedRelease(releaseExtensionId: String): Option[T] = {
    val release = releaseService.findByIdInArchive(releaseIdFrom(releaseExtensionId))
    release.getProperty[JList[T]]("extensions").asScala
      .find(extension => getName(extension.getId) == getName(releaseExtensionId))
  }

  def createDefaultReleaseExtension(releaseExtensionId: String): T

}
