package com.xebialabs.xlrelease.risk.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, Type}
import com.xebialabs.deployit.repository.ItemAlreadyExistsException
import com.xebialabs.xlrelease.domain.events.{ConfigurationCopiedEvent, ConfigurationCreatedEvent, ConfigurationDeletedEvent}
import com.xebialabs.xlrelease.events.EventBus
import com.xebialabs.xlrelease.repository.RetryTitleGenerator.getNextTitle
import com.xebialabs.xlrelease.repository.query.ReleaseBasicData
import com.xebialabs.xlrelease.repository.{ConfigurationRepository, Ids}
import com.xebialabs.xlrelease.risk.domain.RiskProfile
import com.xebialabs.xlrelease.risk.domain.events.RiskProfileUpdated
import com.xebialabs.xlrelease.risk.domain.riskassessors.RiskAssessor
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

@Service
class RiskProfileService @Autowired()(ciIdService: CiIdService,
                                      configurationRepository: ConfigurationRepository,
                                      val eventBus: EventBus) extends Logging {

  @Timed
  def create(riskProfile: RiskProfile): RiskProfile = {
    checkDuplicate(riskProfile.getTitle)
    riskProfile.setId(generateRiskProfileId)
    configurationRepository.create(riskProfile)
    eventBus.publish(ConfigurationCreatedEvent(riskProfile))
    riskProfile
  }

  @Timed
  def exists(riskProfileId: String): Boolean = configurationRepository.exists(riskProfileId)

  @Timed
  def findById(riskProfileId: String): RiskProfile = configurationRepository.read[RiskProfile](riskProfileId)

  @Timed
  def existsByTitle(title: String): Boolean = configurationRepository.existsByTypeAndTitle[RiskProfile](Type.valueOf(classOf[RiskProfile]), title)

  @Timed
  def findByIdOrDefault(riskProfileId: String): RiskProfile = {
    try {
      configurationRepository.read[RiskProfile](riskProfileId)
    } catch {
      case _: NotFoundException => generateDefaultRiskProfile
    }
  }

  @Timed
  def findAll(): JList[RiskProfile] = configurationRepository.findAllByType[RiskProfile](Type.valueOf(classOf[RiskProfile]))

  @Timed
  def findByTitle(riskProfileTitle: String): RiskProfile = {
    val profiles = configurationRepository.findAllByTypeAndTitle[RiskProfile](
      Type.valueOf(classOf[RiskProfile]),
      riskProfileTitle
    )
    profiles.size() match {
      case 0 => throw new NotFoundException(s"Could not find risk profile for title [$riskProfileTitle]")
      case _ => profiles.get(0)
    }
  }

  @Timed
  def update(riskProfile: RiskProfile): RiskProfile = {
    val currentRiskProfile = findById(riskProfile.getId)
    checkArgument(!currentRiskProfile.isDefaultProfile, "Default risk profile cannot be updated")
    if (currentRiskProfile.getTitle != riskProfile.getTitle) {
      checkDuplicate(riskProfile.getTitle)
    }
    configurationRepository.update(riskProfile)
    eventBus.publish(RiskProfileUpdated(riskProfile))
    riskProfile
  }

  @Timed
  def delete(riskProfileId: String): Unit = {
    val riskProfile = findById(riskProfileId)
    checkArgument(!riskProfile.isDefaultProfile, "Default risk profile cannot be deleted")
    configurationRepository.delete(riskProfileId)
    eventBus.publish(ConfigurationDeletedEvent(riskProfile))
  }

  @Timed
  def copy(riskProfileId: String): RiskProfile = {
    val riskProfile = findById(riskProfileId)
    var copyRiskProfileTitle = riskProfile.getTitle
    while (existsByTitle(copyRiskProfileTitle)) {
      copyRiskProfileTitle = getNextTitle(copyRiskProfileTitle)
    }
    val copyRiskProfile: RiskProfile = Type.valueOf(classOf[RiskProfile]).getDescriptor.newInstance(generateRiskProfileId)
    copyRiskProfile.setTitle(copyRiskProfileTitle)
    copyRiskProfile.setRiskProfileAssessors(riskProfile.getRiskProfileAssessors)
    configurationRepository.create(copyRiskProfile)
    eventBus.publish(ConfigurationCopiedEvent(riskProfile, copyRiskProfile.getId))
    copyRiskProfile
  }

  @Timed
  def getReferences(riskProfileId: String): JList[ReleaseBasicData] = {
    configurationRepository.getReferenceReleases(riskProfileId)
  }

  def generateDefaultRiskProfile: RiskProfile = {
    val defaultRiskProfile: RiskProfile = Type.valueOf(classOf[RiskProfile]).getDescriptor.newInstance(null)
    DescriptorRegistry.getSubtypes(Type.valueOf(classOf[RiskAssessor])).asScala
      .filter((`type`: Type) => !`type`.getDescriptor.isVirtual)
      .foreach(t => defaultRiskProfile.setValueFor(t, t.getDescriptor.getPropertyDescriptor("score").getDefaultValue.asInstanceOf[Int]))
    defaultRiskProfile
  }

  private[this] def checkDuplicate(title: String): Unit =
    if (existsByTitle(title)) throw new ItemAlreadyExistsException("Risk profile '%s' already exist.", title)

  private[this] def generateRiskProfileId: String = ciIdService.getUniqueId(Type.valueOf(classOf[RiskProfile]), Ids.RISK_PROFILE_CONFIGURATION_ROOT)
}
