import React, { useState } from 'react';
import { DotButton, DotConfirmationDialog } from '@digital-ai/dot-components';
import { remoteRunners } from '../ducks/remote-runners.reducer';
import { RemoteRunner } from '../remote-runners.types';
import { useAppDispatch } from '../../../../../../../../../core/xlr-ui/app/js/hooks';
import { ConnectionDialog } from '../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/common/connection-dialog/connection-dialog.component';
import { RemoteRunnersTable } from './remote-runners-table';
import { ActionToolbarPortalContainer } from '../../../../../../../../../core/xlr-ui/app/features/main-navigation/action-toolbar/components/action-toolbar.portal';

export interface RemoteRunnersTableContainerProps {
    isLoading: boolean;
    onViewChange: () => void;
    runners: RemoteRunner[];
}

const { reloadRemoteRunners, deleteRemoteRunner } = remoteRunners.actions;

export const RemoteRunnersTableContainer = ({ isLoading, onViewChange, runners }: RemoteRunnersTableContainerProps) => {
    const dispatch = useAppDispatch();

    const [selectedRunner, setSelectedRunner] = useState<RemoteRunner>();
    const [isEditMode, setEditMode] = useState<boolean>(false);
    const [isDeleteMode, setDeleteMode] = useState<boolean>(false);

    const handleEditClick = (remoteRunner: RemoteRunner) => {
        setSelectedRunner(remoteRunner);
        setEditMode(true);
    };

    const handleCloseEditModal = () => {
        setEditMode(false);
    };

    const handleDeleteClick = (remoteRunner: RemoteRunner) => {
        setSelectedRunner(remoteRunner);
        setDeleteMode(true);
    };

    const handleCloseDeleteModal = () => {
        setDeleteMode(false);
    };

    const handleDeleteSubmit = () => {
        if (selectedRunner) {
            dispatch(deleteRemoteRunner(selectedRunner));
            setDeleteMode(false);
        }
    };

    return (
        <>
            <ActionToolbarPortalContainer>
                <DotButton onClick={onViewChange}>Add remote runner</DotButton>
            </ActionToolbarPortalContainer>
            {selectedRunner && isEditMode && (
                <ConnectionDialog
                    configurationSaveAction={reloadRemoteRunners}
                    connectionId={selectedRunner.id}
                    hiddenProperties={['available', 'publicKey', 'encryptJobData']}
                    isNewInstance={false}
                    keyOrName={selectedRunner.id}
                    onCancel={handleCloseEditModal}
                    onSave={handleCloseEditModal}
                    referencedType="xlRelease.RemoteJobRunner"
                />
            )}
            {selectedRunner && isDeleteMode && (
                <DotConfirmationDialog
                    message={
                        <>
                            Are you sure you want to delete <strong>{selectedRunner.runnerName}</strong> remote runner?
                        </>
                    }
                    onCancel={handleCloseDeleteModal}
                    onSubmit={handleDeleteSubmit}
                    open={isDeleteMode}
                    title="Delete remote runner"
                />
            )}
            <RemoteRunnersTable isLoading={isLoading} onDeleteClick={handleDeleteClick} onEditClick={handleEditClick} runners={runners} />
        </>
    );
};
