package com.xebialabs.xlrelease.runner.impl

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.Configuration
import com.xebialabs.xlrelease.domain.runner.RemoteJobRunner
import com.xebialabs.xlrelease.domain.runner.RemoteJobRunner.DEFAULT_REMOTE_JOB_RUNNER_CAPABILITY
import com.xebialabs.xlrelease.repository.Ids.{CUSTOM_CONFIGURATION_ROOT, SEPARATOR}
import com.xebialabs.xlrelease.runner.api.v1.RunnerRegistrationResource.{RegisterRequest, RegisterResponse}
import com.xebialabs.xlrelease.runner.service.{RunnerRegistrationService, RunnerTokenService}
import com.xebialabs.xlrelease.scheduler.domain.LocalJobRunner.DEFAULT_LOCAL_JOB_RUNNER_CAPABILITY
import com.xebialabs.xlrelease.service.ConfigurationService
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class RunnerRegistrationServiceImpl(runnerTokenService: RunnerTokenService,
                                    configurationService: ConfigurationService) extends RunnerRegistrationService {

  @IsTransactional
  @Timed
  override def register(registerRequest: RegisterRequest): RegisterResponse = {
    // All remote runner should have default capabilities as "remote". Remote runners can't have "local" as capability which is reserved for LocalRunners.
    val runnerCapabilities: Set[String] = registerRequest.capabilities ++ Set(DEFAULT_REMOTE_JOB_RUNNER_CAPABILITY) -- Set(DEFAULT_LOCAL_JOB_RUNNER_CAPABILITY)

    val itemId = runnerId(registerRequest.uuid)
    val runner = if (configurationService.exists(itemId)) {
      val existingRunner = configurationService.read(itemId).asInstanceOf[RemoteJobRunner]
      if (existingRunner.getRunnerName() != registerRequest.name || existingRunner.getVersion() != registerRequest.version) {
        existingRunner.setRunnerName(registerRequest.name)
        existingRunner.setVersion(registerRequest.version)
      }
      existingRunner.setTokenExpiryDate(registerRequest.tokenExpiryDate)
      configurationService.createOrUpdate(existingRunner)
      existingRunner
    } else {
      val newRunner = Type.valueOf(classOf[RemoteJobRunner]).getDescriptor.newInstance(itemId).asInstanceOf[RemoteJobRunner]
      newRunner.setTitle(registerRequest.name)
      newRunner.setRunnerName(registerRequest.name)
      newRunner.setVersion(registerRequest.version)
      newRunner.setCapabilities(runnerCapabilities.asJava)
      newRunner.setPublicKey(registerRequest.publicKey)
      newRunner.setTokenExpiryDate(registerRequest.tokenExpiryDate)
      configurationService.createOrUpdate(newRunner)
      newRunner
    }
    val runnerToken = runnerTokenService.createOrUpdateRunnerToken(runner.getId(), registerRequest.tokenExpiryDate)
    val encryptedToken = ContextDataHelper.encryptData(runner, runnerToken.token)
    RegisterResponse(runner.getId(), runner.isEnabled(), runner.capacity, runner.evictionTime, runner.encryptJobData, encryptedToken)
  }

  private def runnerId(uuid: String): String = {
    val parentId = CUSTOM_CONFIGURATION_ROOT
    val typePrefix = Type.valueOf(classOf[Configuration]).getName
    val uuidString = uuid.filterNot(_ == '-')
    s"$parentId$SEPARATOR$typePrefix$uuidString"
  }
}
