package com.xebialabs.xlrelease.runner.impl

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.RUNNER_TOKENS
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, Utils}
import com.xebialabs.xlrelease.runner.domain.RunnerId
import com.xebialabs.xlrelease.runner.repository.RunnerTokenRepository
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component

import java.util.Date

@Component
@IsTransactional
class SqlRunnerTokenRepository(@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                               @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect)
  extends RunnerTokenRepository
    with PersistenceSupport
    with Utils
    with Logging {

  private val FIND_RUNNER_ID_BY_TOKEN =
    s"""
       |SELECT
       |  ${RUNNER_TOKENS.RUNNER_ID}
       | FROM ${RUNNER_TOKENS.TABLE}
       |  WHERE ${RUNNER_TOKENS.TOKEN_HASH} = ?
       |""".stripMargin

  @Timed
  @IsReadOnly
  override def findByToken(tokenHash: String): Option[RunnerId] = {
    findOptional(_.queryForObject(FIND_RUNNER_ID_BY_TOKEN, classOf[String], tokenHash))
  }

  private val RUNNER_ID_EXISTS =
    s"""
       |SELECT COUNT(1)
       | FROM ${RUNNER_TOKENS.TABLE}
       |  WHERE ${RUNNER_TOKENS.RUNNER_ID} = ?
       |""".stripMargin

  @Timed
  @IsReadOnly
  override def exists(runnerId: RunnerId): Boolean = {
    logger.trace(s"Searching runner token for runnerId [$runnerId]")
    val count = jdbcTemplate.queryForObject(RUNNER_ID_EXISTS, classOf[Integer], runnerId)
    count > 0
  }

  private val STMT_INSERT_RUNNER_TOKEN: String =
    s"""
       |INSERT INTO ${RUNNER_TOKENS.TABLE} (
       |    ${RUNNER_TOKENS.RUNNER_ID},
       |    ${RUNNER_TOKENS.TOKEN_HASH},
       |    ${RUNNER_TOKENS.CREATED_DATE},
       |    ${RUNNER_TOKENS.EXPIRY_DATE}
       | ) VALUES (
       |    :${RUNNER_TOKENS.RUNNER_ID},
       |    :${RUNNER_TOKENS.TOKEN_HASH},
       |    :${RUNNER_TOKENS.CREATED_DATE},
       |    :${RUNNER_TOKENS.EXPIRY_DATE}
       | )
       |""".stripMargin

  @Timed
  override def create(runnerId: RunnerId, tokenHash: String, expiryDate: Date): Unit = {
    logger.trace(s"creating runner token for runnerId[$runnerId]")
    val params = Map(
      RUNNER_TOKENS.RUNNER_ID -> runnerId,
      RUNNER_TOKENS.TOKEN_HASH -> tokenHash,
      RUNNER_TOKENS.CREATED_DATE -> new Date(),
      RUNNER_TOKENS.EXPIRY_DATE -> expiryDate
    )
    sqlInsert(STMT_INSERT_RUNNER_TOKEN, params)
  }

  private val STMT_UPDATE_RUNNER_TOKEN: String =
    s"""
       |UPDATE ${RUNNER_TOKENS.TABLE}
       | SET
       |  ${RUNNER_TOKENS.TOKEN_HASH} = :${RUNNER_TOKENS.TOKEN_HASH},
       |  ${RUNNER_TOKENS.UPDATED_DATE} = :${RUNNER_TOKENS.UPDATED_DATE},
       |  ${RUNNER_TOKENS.EXPIRY_DATE} = :${RUNNER_TOKENS.EXPIRY_DATE}
       | WHERE ${RUNNER_TOKENS.RUNNER_ID} = :${RUNNER_TOKENS.RUNNER_ID}
       |""".stripMargin

  @Timed
  override def update(runnerId: RunnerId, tokenHash: RunnerId, expiryDate: Date): Unit = {
    logger.trace(s"updating runner token for runnerId[$runnerId]")
    val params = Map(
      RUNNER_TOKENS.RUNNER_ID -> runnerId,
      RUNNER_TOKENS.TOKEN_HASH -> tokenHash,
      RUNNER_TOKENS.UPDATED_DATE -> new Date(),
      RUNNER_TOKENS.EXPIRY_DATE -> expiryDate
    )
    sqlUpdate(STMT_UPDATE_RUNNER_TOKEN, params, () => _)
  }

  private val STMT_DELETE_RUNNER_TOKEN = s"DELETE FROM ${RUNNER_TOKENS.TABLE} WHERE ${RUNNER_TOKENS.RUNNER_ID} = :${RUNNER_TOKENS.RUNNER_ID}"

  @Timed
  override def delete(runnerId: RunnerId): Unit = {
    logger.trace(s"deleting runner tokens with runnerId[$runnerId]")
    sqlExec(STMT_DELETE_RUNNER_TOKEN, params(RUNNER_TOKENS.RUNNER_ID -> runnerId), _.execute())
  }

  private val FIND_EXPIRY_DATE_BY_TOKEN =
    s"""
       |SELECT
       |  ${RUNNER_TOKENS.EXPIRY_DATE}
       | FROM ${RUNNER_TOKENS.TABLE}
       |  WHERE ${RUNNER_TOKENS.TOKEN_HASH} = ?
       |""".stripMargin

  @Timed
  @IsReadOnly
  override def findExpiryDate(tokenHash: String): Option[Date] = {
    findOptional(_.queryForObject(FIND_EXPIRY_DATE_BY_TOKEN, classOf[Date], tokenHash))
  }
}
