import { SagaIterator } from 'redux-saga';
import { all, call, put, select, takeEvery } from 'redux-saga/effects';
import { remoteRunners, UpdateRemoteRunnerStatePayload } from './remote-runners.reducer';
import remoteRunnersSaga, {
    deleteRemoteRunnerAction,
    loadRemoteRunnersAction,
    reloadRemoteRunnersAction,
    updateRemoteRunnerStateAction,
    withLoadingState,
} from './remote-runners.saga';
import { httpDELETE, httpGET, httpPUT } from '@xlr-ui/app/features/common/services/http';
import { RemoteRunner } from '../remote-runners.types';
import { remoteRunnerMock } from '../__mocks__/remote-runners.mocks';
import { ConfigurationCallbackPayload } from '@xlr-ui/app/features/configuration/types';
import { PayloadAction } from '@reduxjs/toolkit';
import { getRemoteRunnersSelector } from './remote-runners.selectors';

const { loadRemoteRunners, setRemoteRunners, reloadRemoteRunners, updateRemoteRunnerState, deleteRemoteRunner } = remoteRunners.actions;

describe('remoteRunnersSaga', () => {
    it('should yield all effects', () => {
        const gen: SagaIterator = remoteRunnersSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeEvery(loadRemoteRunners, loadRemoteRunnersAction),
                takeEvery(reloadRemoteRunners, reloadRemoteRunnersAction),
                takeEvery(updateRemoteRunnerState, updateRemoteRunnerStateAction),
                takeEvery(deleteRemoteRunner, deleteRemoteRunnerAction),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('loadRemoteRunnersAction', () => {
        it('should handle loadRemoteRunners action', () => {
            const gen: SagaIterator = loadRemoteRunnersAction();
            expect(gen.next().value).toStrictEqual(
                call(withLoadingState, call(httpGET, `api/v1/config/byTypeAndTitle?configurationType=xlrelease.RemoteJobRunner`)),
            );
            expect(gen.next({ data: [remoteRunnerMock] }).value).toStrictEqual(put(setRemoteRunners([remoteRunnerMock])));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('reloadRemoteRunnersAction', () => {
        it('should handle reloadRemoteRunners action', () => {
            const action: PayloadAction<ConfigurationCallbackPayload> = {
                payload: {
                    id: '',
                    isUpdate: false,
                },
                type: '',
            };
            const gen: SagaIterator = reloadRemoteRunnersAction(action);
            expect(gen.next().value).toStrictEqual(call(loadRemoteRunnersAction));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('updateRemoteRunnerStateAction', () => {
        it('should handle updateRemoteRunnerState action', () => {
            const action: PayloadAction<UpdateRemoteRunnerStatePayload> = {
                type: '',
                payload: {
                    id: remoteRunnerMock.id,
                    enable: false,
                },
            };
            const updatedRunner = {
                ...remoteRunnerMock,
                enabled: false,
            };

            const gen: SagaIterator = updateRemoteRunnerStateAction(action);
            expect(gen.next().value).toStrictEqual(call(httpGET, `api/v1/config/${action.payload.id}`));
            expect(gen.next({ data: updatedRunner }).value).toStrictEqual(call(httpPUT, `api/v1/config/${action.payload.id}`, updatedRunner));

            const runners: RemoteRunner[] = [remoteRunnerMock];
            expect(gen.next({ data: runners }).value).toStrictEqual(select(getRemoteRunnersSelector));

            const updatedRunners: RemoteRunner[] = [updatedRunner];
            expect(gen.next(updatedRunners).value).toStrictEqual(put(setRemoteRunners(updatedRunners)));

            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('deleteRemoteRunnerAction', () => {
        it('should handle deleteRemoteRunner action', () => {
            const action: PayloadAction<RemoteRunner> = {
                payload: remoteRunnerMock,
                type: '',
            };
            const gen: SagaIterator = deleteRemoteRunnerAction(action);
            expect(gen.next({ data: remoteRunnerMock }).value).toStrictEqual(call(httpDELETE, `api/v1/config/${action.payload.id}`));

            const runners: RemoteRunner[] = [remoteRunnerMock];
            expect(gen.next({ data: runners }).value).toStrictEqual(select(getRemoteRunnersSelector));

            const updatedRunners: RemoteRunner[] = [];
            expect(gen.next(updatedRunners).value).toStrictEqual(put(setRemoteRunners(updatedRunners)));

            expect(gen.next().done).toBeTruthy();
        });
    });
});
