package com.xebialabs.xlrelease.runner.impl

import com.fasterxml.jackson.annotation.JsonInclude
import com.fasterxml.jackson.core.`type`.TypeReference
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.module.scala.DefaultScalaModule
import com.xebialabs.xlrelease.builder.TaskBuilder.newRemoteExecution
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.distributed.events.DistributedXLReleaseEvent
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.runner.impl.RunnerScriptService.{CommandResponseKey, ErrorMessageKey}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.newEncryptingConverter
import com.xebialabs.xlrelease.service.BroadcastService
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import java.util
import scala.concurrent.{Await, Promise, TimeoutException}
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}


object RunnerScriptService {
  val CommandResponseKey = "commandResponse"
  val ErrorMessageKey = "errorMessage"
}

trait RunnerScriptService {
  def executeScript[T](script: String, inputParameters: util.Map[String, AnyRef], typeReference: TypeReference[T]): T

  def executeScript[T](script: String, inputParameters: util.Map[String, AnyRef]): Vector[T]

  def executeScriptForObject[T](script: String, inputParameters: util.Map[String, AnyRef]): T

  def scriptExecuted(scriptExecutionId: String, payload: util.Map[String, AnyRef]): Unit
}

@Service
@EventListener
class RunnerScriptServiceImpl(xlrConfig: XlrConfig,
                              jobRunnerService: JobRunnerService,
                              broadcastService: BroadcastService) extends RunnerScriptService with Logging {
  private val responsePromises = scala.collection.concurrent.TrieMap[String, Promise[ContainerExecutionResultEvent]]()

  private def objectMapper: ObjectMapper = {
    val mapper = new ObjectMapper()
    mapper.setSerializationInclusion(JsonInclude.Include.ALWAYS)
    mapper.registerModule(DefaultScalaModule)
    mapper
  }

  override def executeScript[T](script: String, inputParameters: util.Map[String, AnyRef], typeReference: TypeReference[T]): T = {
    val scriptResponse = executeOnContainer(script, inputParameters).payload.asScala.toMap

    scriptResponse.get(CommandResponseKey) match {
      case Some(commandResponse) =>
        Try(objectMapper.readValue(objectMapper.writeValueAsString(commandResponse), typeReference)) match {
          case Success(result) => result
          case Failure(exception) => throw ContainerScriptException(s"Error deserializing container $script response: $exception")
        }
      case None if scriptResponse.contains(ErrorMessageKey) =>
        val error = objectMapper.readValue(objectMapper.writeValueAsString(scriptResponse(ErrorMessageKey)), classOf[String])
        logger.warn(s"Container script $script failed with error: $error")
        throw ContainerScriptException(s"Container script $script failed with error: $error")
      case _ =>
        logger.error(s"Failed to process container script $script result")
        throw ContainerScriptException(s"Failed to process container script $script result")
    }
  }

  override def executeScript[T](script: String, inputParameters: util.Map[String, AnyRef]): Vector[T] = {
    val scriptResponse = executeOnContainer(script, inputParameters).payload.asScala.toMap
    scriptResponse.get(CommandResponseKey) match {
      case Some(commandResponse: util.List[_]) =>
        commandResponse.asScala.map { resultItem =>
          val nodeString = objectMapper.writeValueAsString(resultItem)
          Try(CiSerializerHelper.deserialize(nodeString, null, newEncryptingConverter()).asInstanceOf[T]) match {
            case Success(result) =>
              logger.debug(s"Container script $script executed successfully")
              result
            case Failure(exception) => throw ContainerScriptException(s"Error deserializing container $script response: $exception")
          }
        }.toVector
      case None if scriptResponse.contains(ErrorMessageKey) =>
        val error = objectMapper.readValue(objectMapper.writeValueAsString(scriptResponse(ErrorMessageKey)), classOf[String])
        logger.warn(s"Container script $script failed with error: $error")
        throw ContainerScriptException(s"Container script $script failed with error: $error")
      case _ =>
        logger.error(s"Failed to process container script $script result")
        throw ContainerScriptException(s"Failed to process container script $script result")
    }
  }

  override def executeScriptForObject[T](script: String, inputParameters: util.Map[String, AnyRef]): T = {
    val scriptResponse = executeOnContainer(script, inputParameters).payload.asScala.toMap
    scriptResponse.get(CommandResponseKey) match {
      case Some(commandResponse) =>
        val nodeString = objectMapper.writeValueAsString(commandResponse)
        Try(CiSerializerHelper.deserialize(nodeString, null, newEncryptingConverter()).asInstanceOf[T]) match {
          case Success(result) =>
            logger.debug(s"Container script $script executed successfully")
            result
          case Failure(exception) => throw ContainerScriptException(s"Error deserializing container $script response: $exception")
        }
      case None if scriptResponse.contains(ErrorMessageKey) =>
        logger.warn(s"Container script $script failed with error: ${scriptResponse(ErrorMessageKey)}")
        val error = objectMapper.readValue(objectMapper.writeValueAsString(scriptResponse(ErrorMessageKey)), classOf[String])
        throw ContainerScriptException(s"Container script $script failed with error: $error")
      case _ =>
        logger.error(s"Failed to process container script $script result")
        throw ContainerScriptException(s"Failed to process container script $script result")
    }
  }

  private def executeOnContainer(script: String, inputParameters: util.Map[String, AnyRef]): ContainerExecutionResultEvent = {
    val scriptTask = newRemoteExecution(script)
      .withInputParameters(inputParameters)
      .build()

    val runner = jobRunnerService.findScriptExecutor() match {
      case Some(runner) => runner
      case None =>
        logger.warn(s"Cannot find active remote runner for executing script '$script'")
        throw ContainerScriptException(s"Cannot find active remote runner for executing script '$script'")
    }

    val scriptExecutionId = runner.execute(scriptTask)
    val responsePromise = Promise[ContainerExecutionResultEvent]()
    responsePromises.put(scriptExecutionId, responsePromise)

    try {
      Await.result(responsePromise.future, xlrConfig.timeouts.releaseActionResponse)
    } catch {
      case e: TimeoutException =>
        logger.error(s"Execution of script '$script' timed out")
        throw ContainerScriptException(s"Execution of script '$script' timed out: ${e.getMessage}")
    }
  }

  override def scriptExecuted(scriptExecutionId: String, payload: util.Map[String, AnyRef]): Unit = {
    val response = ContainerExecutionResultEvent(scriptExecutionId, payload)
    logger.debug(s"Sending execution response for $scriptExecutionId")
    broadcastService.broadcast(response, publishEventOnSelf = true)
  }

  @AsyncSubscribe
  def onContainerExecutionResultEvent(response: ContainerExecutionResultEvent): Unit = {
    logger.debug(s"Received execution response for ${response.scriptExecutionId}")
    responsePromises.get(response.scriptExecutionId).foreach { promise =>
      try {
        promise.success(response)
      } catch {
        case e: Exception =>
          logger.error(s"Failed to complete promise for ${response.scriptExecutionId}: ${e.getMessage}")
      } finally {
        responsePromises.remove(response.scriptExecutionId)
      }
    }
  }
}

case class ContainerScriptException(msg: String) extends Exception(msg)

// we want to reduce the size of event by not sending the payload in future
case class ContainerExecutionResultEvent(scriptExecutionId: String, payload: util.Map[String, AnyRef]) extends DistributedXLReleaseEvent
