package com.xebialabs.xlrelease.runner.impl

import com.xebialabs.xlrelease.actors.XlrPersistentActor
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.runner.domain._
import com.xebialabs.xlrelease.runner.impl.RunnerControlChannelActor.SendCommand
import com.xebialabs.xlrelease.runner.impl.RunnerProxyActor._
import com.xebialabs.xlrelease.support.pekko.spring.SpringActor
import org.apache.pekko.actor.{ActorRef, PoisonPill}
import org.apache.pekko.persistence._

import java.util.UUID
import scala.collection.mutable

@SpringActor
class RunnerProxyActor(runnerId: RunnerId) extends XlrPersistentActor {
  private var state: RunnerProxyState = Initial(runnerId, Map())

  private val openChannels: mutable.Set[ActorRef] = mutable.Set()

  private val snapshotAfter: Int = XlrConfig.getInstance.pekko.snapshotAfter

  private val keepNrOfBatches: Int = XlrConfig.getInstance.pekko.keepNrOfBatches

  private var runnerHealth: mutable.Map[RunnerId, Long] = mutable.Map.empty

  override def persistenceId: String = self.path.name

  override def receiveRecover: Receive = {
    case evt: RunnerProxyEvent =>
      state = state.applyEvent(evt)
    case SnapshotOffer(_, snapshot: RunnerProxyState) =>
      state = snapshot
    case RecoveryCompleted =>
      log.debug(s"Recovery completed for $persistenceId. State is $state")
    case msg =>
      log.warning(s"Can't recover message [$msg]")
  }

  override def receiveCommand: Receive = handleRunnerProxyCommand orElse handleAliveMessage orElse handleInternalMessage orElse handleUnknownMessage

  def handleRunnerProxyCommand: Receive = {
    case cmd: RunnerProxyCommand => handleCommand(cmd)
  }

  def handleAliveMessage: Receive = {
    case RunnerAlive(runnerId, timestamp) =>
      runnerHealth.put(runnerId, timestamp)
    case LastRunnerTime(runnerId) =>
      sender() ! LastRunnerTimeResponse(runnerId, runnerHealth.get(runnerId))
  }


  def handleUnknownMessage: Receive = {
    case unknown => log.error(s"Received unknown command $unknown")
  }

  def handleInternalMessage: Receive = {
    case SaveSnapshotSuccess(metadata) =>
      log.debug("Snapshot saved successfully")
      val sequenceNr = metadata.sequenceNr - keepNrOfBatches * snapshotAfter
      if (sequenceNr > 0) {
        deleteMessages(sequenceNr)
      }

    case SaveSnapshotFailure(_, reason) =>
      log.warning("Snapshot failure: [{}]", reason.getMessage)

    case DeleteMessagesSuccess(toSequenceNr) =>
      val deleteTo = toSequenceNr - 1
      log.debug(s"Messages to [$toSequenceNr] deleted successfully. Deleting snapshots.")
      deleteSnapshots(SnapshotSelectionCriteria(maxSequenceNr = deleteTo))

    case DeleteMessagesFailure(reason, toSequenceNr) =>
      log.warning("Messages to [{}] deletion failure: [{}]", toSequenceNr, reason.getMessage)

    case DeleteSnapshotsSuccess(m) =>
      log.debug("Snapshots matching [{}] deleted successfully", m)

    case DeleteSnapshotsFailure(m, reason) =>
      log.warning("Snapshots matching [{}] deletion failure: [{}]", m, reason.getMessage)

    case PoisonPill =>
      context.stop(self)
  }

  private def handleCommand(command: RunnerProxyActor.RunnerProxyCommand): Unit = command match {
    case RunnerProxyActor.NewCommand(runnerId, command) =>
      val commandId = UUID.randomUUID().toString
      persist(CommandAdded(PersistedCommand(commandId, command)))(handleEvent)
    case RunnerProxyActor.ConfirmCommand(runnerId, commandId) =>
      persist(CommandConfirmed(commandId))(handleEvent)
    case RunnerProxyActor.AddChannel(runnerId, controlChannel) =>
      openChannels.add(controlChannel)
      // publish all commands in the state
      state.commands.foreach {
        case (commandId, command) => controlChannel ! SendCommand(runnerId, command)
      }
    case RunnerProxyActor.RemoveChannel(runnerId, controlChannel) =>
      openChannels.remove(controlChannel)
  }

  private def handleEvent(event: RunnerProxyEvent): Unit = {
    state = state.applyEvent(event)
    saveSnapshotWhenNeeded()
    event match {
      case CommandAdded(command) =>
        openChannels.foreach(_ ! SendCommand(runnerId, command))
      case CommandConfirmed(_) =>
        // do nothing as command will be removed from internal state in applyEvent
        ()
    }
  }

  private def saveSnapshotWhenNeeded(): Unit = {
    if (lastSequenceNr % snapshotAfter == 0 && lastSequenceNr != 0) {
      log.debug("Saving snapshot, sequence number [{}]", snapshotSequenceNr)
      saveSnapshot(state)
    }

  }

}

object RunnerProxyActor {

  final val SHARDING_TYPE_NAME = "runner-proxy"

  def actorName(runnerId: RunnerId): String = {
    s"runner-proxy-${runnerId.shortId()}"
  }

  sealed trait RunnerProxyState {
    def applyEvent(event: RunnerProxyEvent): RunnerProxyState

    def commands: Map[String, PersistedCommand]
  }

  case class Initial(runnerId: RunnerId, commands: Map[String, PersistedCommand]) extends RunnerProxyState {
    override def applyEvent(event: RunnerProxyEvent): RunnerProxyState = event match {
      case CommandAdded(persistedCommand) =>
        if (null != persistedCommand && null != persistedCommand.command && persistedCommand.command.commandConfirmationRequired) {
          copy(commands = commands + (persistedCommand.commandId -> persistedCommand))
        } else {
          this
        }
      case CommandConfirmed(commandId) =>
        copy(commands = commands - commandId)
    }
  }

  // commands
  sealed trait RunnerProxyCommand {
    def runnerId: RunnerId
  }

  case class NewCommand(runnerId: RunnerId, command: RunnerControlCommand) extends RunnerProxyCommand

  case class ConfirmCommand(runnerId: RunnerId, commandId: String) extends RunnerProxyCommand
  case class AddChannel(runnerId: RunnerId, controlChannel: ActorRef) extends RunnerProxyCommand

  case class RemoveChannel(runnerId: RunnerId, controlChannel: ActorRef) extends RunnerProxyCommand

  // events
  sealed trait RunnerProxyEvent

  case class CommandAdded(persistedCommand: PersistedCommand) extends RunnerProxyEvent

  case class CommandConfirmed(commandId: String) extends RunnerProxyEvent

  sealed trait RunnerHealth {
    def runnerId: RunnerId
  }

  case class RunnerAlive(runnerId: RunnerId, timestamp: Long) extends RunnerHealth

  case class LastRunnerTime(runnerId: RunnerId) extends RunnerHealth
  case class LastRunnerTimeResponse(runnerId: RunnerId, timestamp: Option[Long]) extends RunnerHealth
}
