package com.xebialabs.xlrelease.webhooks.listeners

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlplatform.webhooks.events.domain.{EventConsumer, EventSource}
import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{AsyncSubscribe, SynchronizedSubscribe, XLReleaseEventListener}
import com.xebialabs.xlrelease.webhooks.registry.SubscriberRegistry
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired

class BaseConsumerConfigurationEventListener extends XLReleaseEventListener with Logging {
  @Autowired
  private var subscriberRegistry: SubscriberRegistry = _

  private lazy val baseConsumerType: Type = Type.valueOf(classOf[EventConsumer])

  @AsyncSubscribe
  def onConfigurationCreated(event: ConfigurationCreatedEvent): Unit = {
    onEventConsumerConfig(event.conf)
      .filter(_.consumerEnabled)
      .foreach(subscribeConf)
  }

  @AsyncSubscribe
  def onConfigurationUpdated(event: ConfigurationUpdatedEvent): Unit = {
    onEventConsumerConfig(event.updated).foreach { conf =>
      unsubscribeConf(conf)
      if (conf.consumerEnabled) {
        subscribeConf(conf)
      }
    }
  }

  @SynchronizedSubscribe
  def onConfigurationDelete(event: ConfigurationDeletedEvent): Unit = {
    onEventConsumerConfig(event.conf).foreach(unsubscribeConf)
  }

  def onEventConsumerConfig(conf: BaseConfiguration): Option[EventConsumer] = {
    logger.trace(s"config event on ${conf.getType}/${conf.getId}")
    if (conf.getType.instanceOf(baseConsumerType)) {
      Some(conf.asInstanceOf[EventConsumer])
    } else {
      logger.trace(s"types mismatch: ${conf.getType} not instanceOf $baseConsumerType")
      None
    }
  }

  def subscribeConf(conf: EventConsumer): Unit = {
    logger.debug(s"subscribeConf: ${conf.consumerId} <- ${conf.eventSource}")
    Option(conf.eventSource)
      .flatMap((source: EventSource) => Option(source.getId))
      .foreach(subscriberRegistry.subscribe(_, conf))
  }

  def unsubscribeConf(conf: EventConsumer): Unit = {
    logger.debug(s"unsubscribeConf: ${conf.consumerId}")
    subscriberRegistry.remove(conf.consumerId)
  }
}

