package com.xebialabs.xlrelease.webhooks.registry

import com.xebialabs.xlplatform.webhooks.events.domain.{EventConsumer, EventSource}
import com.xebialabs.xlrelease.domain.distributed.events.DistributedXLReleaseEvent
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.service.BroadcastService
import com.xebialabs.xlrelease.webhooks.registry.SubscriberRegistryService.{RemoveConsumer, SubscribeConsumer, SubscriberRegistryMessage, UnsubscribeConsumer}
import org.springframework.stereotype.Component

@Component
@EventListener
class SubscriberRegistryService(broadcastService: BroadcastService)
  extends SubscriberRegistry {

  override def subscribe(sourceId: EventSource.ID, consumerConfig: EventConsumer): Unit = {
    super.subscribe(sourceId, consumerConfig)
    // TODO check that it is ok not to publish subscriberRegistry messages on the originating server
    broadcastService.broadcast(SubscribeConsumer(sourceId, consumerConfig), false)
  }

  override def unsubscribe(sourceId: EventSource.ID, consumerId: EventConsumer.ID): Unit = {
    super.unsubscribe(sourceId, consumerId)
    broadcastService.broadcast(UnsubscribeConsumer(sourceId, consumerId), false)
  }

  override def remove(consumerId: EventConsumer.ID): Unit = {
    super.remove(consumerId)
    broadcastService.broadcast(RemoveConsumer(consumerId), false)
  }

  @AsyncSubscribe
  def onSubscriberRegistryMessage(event: SubscriberRegistryMessage): Unit = {
    event match {
      case SubscribeConsumer(sourceId, consumerConfig) =>
        super.subscribe(sourceId, consumerConfig)
      case UnsubscribeConsumer(sourceId, consumerId) =>
        super.unsubscribe(sourceId, consumerId)
      case RemoveConsumer(consumerId) =>
        super.remove(consumerId)
    }
  }
}

object SubscriberRegistryService {
  sealed trait SubscriberRegistryMessage extends DistributedXLReleaseEvent

  // TODO consumerConfig should be consumerId in this message - we do not want to (de)serialize whole ConfigurationItems
  case class SubscribeConsumer(sourceId: EventSource.ID, consumerConfig: EventConsumer) extends SubscriberRegistryMessage

  case class UnsubscribeConsumer(sourceId: EventSource.ID, consumerId: EventConsumer.ID) extends SubscriberRegistryMessage

  case class RemoveConsumer(consumerId: EventConsumer.ID) extends SubscriberRegistryMessage

}
