package com.xebialabs.xlrelease.webhooks.endpoint

import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.plugin.api.validation.{ExtendedValidationContext, ValidationContext, Validator}
import com.xebialabs.xlplatform.webhooks.domain.Endpoint
import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.webhooks.endpoint.EndpointValidator.{invalidCharacter, isPathInvalid}
import com.xebialabs.xlrelease.webhooks.registry.EndpointRegistry
import grizzled.slf4j.Logging

abstract class EndpointValidator[E <: Endpoint with BaseConfigurationItem] extends Validator[E] with Logging {

  def getValidatedPath(ci: E): String

  override def validate(ci: E, context: ValidationContext): Unit = {
    val extendedContext = context.asInstanceOf[ExtendedValidationContext]
    val errorMessage: Option[String] = {
      if (isPathInvalid(ci.getPath)) {
        Some(
          s"""|HTTP endpoint for Webhooks's path contains invalid characters.
              |Invalid characters: ${invalidCharacter.mkString(" ")}""".stripMargin)
      } else {
        val path = getValidatedPath(ci)
        EndpointRegistry.findEndpoint(path) match {
          case Some(endpoint) if !endpoint.equals(ci) =>
            logger.debug(s"validate(${ci. getId}): Found existing HTTP endpoint for Webhooks with path '$path': ${endpoint.sourceId}'")
            endpoint match {
              case baseConfiguration: BaseConfiguration =>
                Some(s"HTTP endpoint for Webhooks path must be unique. Found same path for '${baseConfiguration.getTitle}'")
              case _ =>
                Some(s"HTTP endpoint for Webhooks path must be unique. Found same path for CI ${endpoint}")
            }
          case _ =>
            None
        }
      }
    }
    errorMessage.foreach(extendedContext.error(ci, "path", _))
  }
}

object EndpointValidator {
  lazy val invalidCharacter: Set[Char] = Set(
    '{', '}', '|', '\\', '^', '[', ']', '`',
    ';', '/', '?', ':', '@', '&', '=', '+', '$', ','
  )

  def isPathInvalid(path: String): Boolean = {
    path.exists(invalidCharacter.contains)
  }
}
