'use strict';

angular.module('xlrelease').controller('releaseSummaryController', ['$scope', '$route', 'TasksService', 'FlagService', 'ReleasesService', 'DateService', 'Timeline', 'ColorService', 'SpecialDaysService', function ($scope, $route, TasksService, FlagService, ReleasesService, DateService, Timeline, ColorService, SpecialDaysService) {

    $scope.release = $route.current.locals.releasePromise.data;
    $scope.outgoingDependencies = $route.current.locals.releaseDependenciesPromise.data.outgoingDependencies;
    $scope.incomingDependencies = $route.current.locals.releaseDependenciesPromise.data.incomingDependencies;
    $scope.outgoingReleases = _.pluck($scope.outgoingDependencies, 'release');
    $scope.incomingReleases = _.pluck($scope.incomingDependencies, 'release');

    $scope.flattenedTasks = ReleasesService.getLeafTasks($scope.release);
    $scope.numberOfCompletedTasks = _.filter($scope.flattenedTasks, isTaskCompleted).length;
    $scope.teams = _.chain($scope.flattenedTasks).filter(TasksService.hasTeam).pluck('team').uniq().value();
    $scope.owners = _.chain($scope.flattenedTasks).filter(TasksService.hasOwner).pluck('owner').uniq().value();

    var timelineReleases = _.union($scope.outgoingReleases, [$scope.release], $scope.incomingReleases);

    $scope.timeline = Timeline.init($scope.release, timelineReleases);
    ReleasesService.getReleasesDependenciesByReleaseId(timelineReleases).then(function (releasesDependencies) {
        $scope.releasesDependencies = releasesDependencies;
    });

    computeAlerts();
    SpecialDaysService.loadTimelineSpecialDays($scope.timeline.days);

    function computeAlerts() {
        computeFlaggedItems();
        computeConflicts();
        computeDelayedTasks();
    }

    function computeFlaggedItems() {
        var flaggedItems = [];

        if (FlagService.isReleaseFlagged($scope.release)) {
            flaggedItems.push({flag: {status: $scope.release.releaseFlag.status, comment: $scope.release.releaseFlag.comment}});
        }

        _.each($scope.flattenedTasks, function (task) {
            if (FlagService.isTaskFlagged(task)) {
                flaggedItems.push(task);
            }
        });

        $scope.flaggedItems = flaggedItems;
    }

    function computeConflicts() {
        var conflicts = [];

        _.each($scope.outgoingDependencies, function (dependency) {
            if (dependency.inConflict) {
                conflicts.push(dependency);
            }
        });

        $scope.conflicts = conflicts;
    }

    function computeDelayedTasks() {
        var delayedTasks = [];

        _.each($scope.flattenedTasks, function (task) {
            if (TasksService.isTaskDelayed(task)) {
                delayedTasks.push(task);
            }
        });

        $scope.delayedTasks = delayedTasks;
    }

    function isTaskCompleted(task) {
        return TasksService.isTaskCompleted(task) || TasksService.isTaskSkipped(task);
    }

    $scope.firstDayOfWeek = function (day) {
        return day.fullDate.day() === 0;
    };

    $scope.firstDayOfMonth = function (day) {
        return day.fullDate.date() === 1;
    };

    $scope.isInConflict = function (releaseId) {
        return ReleasesService.isInConflict(releaseId, $scope.releasesDependencies);
    };

    $scope.isTaskPending = TasksService.isTaskPending;
    $scope.isTaskFailed = TasksService.isTaskFailed;
    $scope.isTaskInProgress = TasksService.isTaskInProgress;
    $scope.isTaskFlagged = FlagService.isTaskFlagged;
    $scope.isTaskDelayed = TasksService.isTaskDelayed;
    $scope.isReleaseInProgress = ReleasesService.isReleaseInProgress;
    $scope.isReleaseFailing = ReleasesService.isReleaseFailing;
    $scope.isReleasePaused = ReleasesService.isReleasePaused;
    $scope.isReleaseFailed = ReleasesService.isReleaseFailed;
    $scope.isReleasePlanned = ReleasesService.isReleasePlanned;
    $scope.isReleaseAborted = ReleasesService.isReleaseAborted;
    $scope.isAssignedToUser = TasksService.isAssignedToUser;
    $scope.isAssignedToTeam = TasksService.isAssignedToTeam;
    $scope.computeDayColor = ColorService.computeDayColor;
    $scope.computePhaseColor = ColorService.computePhaseColor;
}]);
