angular.module('xlrelease').factory('TaskPolling', ['Timeout', '$q', 'Backend', 'TasksService', function (Timeout, $q, Backend, TasksService) {

    var POLLING_INTERVAL_IN_MS = 1000;

    function startTaskPolling(tasks, scope) {
        scope.$broadcast('STOP_TASK_POLLING');

        var pollingContext = {
            taskUpdateDeferred : $q.defer(),
            timeoutPromise: null
        };

        var taskUpdatePromise = initPromise(pollingContext);

        var pollableTasks = getPollableTasks(tasks);
        if (pollableTasks.length !== 0) {
            registerTimeoutCancellation(scope, pollingContext);
            poll(pollableTasks, pollingContext);
        }

        return taskUpdatePromise;
    }

    function initPromise(pollingContext) {
        var taskUpdatePromise = pollingContext.taskUpdateDeferred.promise;

        taskUpdatePromise.onTaskUpdate = function (callback) {
            taskUpdatePromise.then(callback);
        };

        return taskUpdatePromise;
    }

    function getPollableTasks(tasks) {
        return _.filter(tasks, function (task) {
            return TasksService.isTaskInProgress(task) || TasksService.isTaskFailed(task) || TasksService.isTaskPending(task);
        });
    }

    function registerTimeoutCancellation(scope, pollingContext) {
        scope.$on('$destroy', function () {
            Timeout.cancel(pollingContext.timeoutPromise);
        });
        scope.$on('STOP_TASK_POLLING', function () {
            Timeout.cancel(pollingContext.timeoutPromise);
        });
    }

    function poll(tasks, pollingContext) {
        pollingContext.timeoutPromise = Timeout(function pollTaskStatus() {
            Backend.post('tasks/poll?' + getRefreshParams(tasks)).success(function (serverTasks) {
                updateTaskInfosFromServer(tasks, serverTasks);

                if (isAnyServerTaskUpdated(tasks, serverTasks)) {
                    pollingContext.taskUpdateDeferred.resolve();
                } else {
                    poll(tasks, pollingContext);
                }
            });
        }, POLLING_INTERVAL_IN_MS);
    }
    
    function getRefreshParams(tasks) {
        return _.map(tasks, function (task) {
            return 'id=' + task.id;
        }).join('&');
    }

    function updateTaskInfosFromServer(tasks, serverTasks) {
        _.map(tasks, function (task) {
            if (TasksService.isDeployitTask(task)) {
                var serverTask = getAssociatedTask(task, serverTasks);
                task.currentStep = serverTask.currentStep;
                task.totalStep = serverTask.totalStep;
                task.currentStepTitle = serverTask.currentStepTitle;
                task.deploymentStatus = serverTask.status;
            }
        })
    }

    function getAssociatedTask(task, serverTasks) {
        return _.find(serverTasks, function (candidate) {
            return task.id === candidate.id;
        });
    }

    function isAnyServerTaskUpdated(tasks, serverTasks) {
        return _.any(tasks, function (task) {
            var serverTask = getAssociatedTask(task, serverTasks);
            return task.status !== serverTask.status
        });
    }

    return {
        startTaskPolling: startTaskPolling
    }
}]);
