package com.xebialabs.xlrelease.auth.oidc.web;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Writer;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.web.filter.GenericFilterBean;
import com.samskivert.mustache.Mustache;
import com.samskivert.mustache.Template;

import static com.xebialabs.xlrelease.auth.oidc.config.OpenIdConnectConfig.OIDC_LOGIN;
import static com.xebialabs.xlrelease.auth.oidc.config.OpenIdConnectConfig.OIDC_PROCESSING_URL;

public class OpenIdConnectRetainAnchorFilter extends GenericFilterBean {

    private final String sameSite;

    private Boolean secureCookie;

    public OpenIdConnectRetainAnchorFilter(String sameSite) {
        this.sameSite = sameSite;
    }

    @Override
	public void doFilter(ServletRequest request, ServletResponse response,
	                     FilterChain chain) throws IOException, ServletException {
		if (response instanceof HttpServletResponse) {
			response = new RedirectResponseWrapper((HttpServletRequest)request, (HttpServletResponse)response);
		}
		chain.doFilter(request, response);
	}

    public void setSecureCookie(final Boolean secureCookie) {
        this.secureCookie = secureCookie;
    }

    /**
	 * HttpServletResponseWrapper that replaces the redirect by appropriate Javascript code.
	 */
	private class RedirectResponseWrapper extends HttpServletResponseWrapper {

        public static final String OPENID_CONNECT_RETAINED_ANCHOR = "XL_RELEASE_OPENID_CONNECT_RETAINED_ANCHOR";
        private final Resource restoreAnchorRedirectPageResource;
		private final Resource storeAnchorRedirectPageResource;
        private final Mustache.Compiler compiler;
        private final HttpServletRequest request;
        private final AntPathRequestMatcher oidcLoginMatcher;
        private final AntPathRequestMatcher oidcLoginSuccessMatcher;

        public RedirectResponseWrapper(HttpServletRequest request, HttpServletResponse response) {
            super(response);
            this.request = request;
            this.compiler = Mustache.compiler()
					.escapeHTML(false)
					.emptyStringIsFalse(true)
					.defaultValue("");
			this.restoreAnchorRedirectPageResource = new ClassPathResource("com/xebialabs/xlrelease/auth/oidc/web/restore-anchor-redirect-page.mustache");
			this.storeAnchorRedirectPageResource = new ClassPathResource("com/xebialabs/xlrelease/auth/oidc/web/store-anchor-redirect-page.mustache");
            this.oidcLoginMatcher = new AntPathRequestMatcher(OIDC_LOGIN);
            this.oidcLoginSuccessMatcher = new AntPathRequestMatcher(OIDC_PROCESSING_URL);
		}

		@Override
		public void sendRedirect(String location) throws IOException {
            if (oidcLoginMatcher.matches(request) && !requestHasAuthorizationParameters()) {
                renderTemplate(location, this.storeAnchorRedirectPageResource);
                return;
            } else if (oidcLoginSuccessMatcher.matches(request) && requestHasAuthorizationParameters() && hasCookie(OPENID_CONNECT_RETAINED_ANCHOR)) {
                renderTemplate(location, this.restoreAnchorRedirectPageResource);
                return;
            }
            super.sendRedirect(location);
		}

		private boolean requestHasAuthorizationParameters() {
            return request.getParameter("state") != null && request.getParameter("code") != null;
        }

		private void renderTemplate(String location, Resource resource) throws IOException {
            HttpServletResponse response = (HttpServletResponse)getResponse();
            Template template = compiler.compile(new InputStreamReader(resource.getInputStream()));
            String cookieSecureString;
            if ((secureCookie != null && secureCookie) || request.isSecure()) {
                cookieSecureString = "secure;";
            } else {
                cookieSecureString = "";
            }

            Map<String, String> scope = new HashMap<>();
            scope.put("location", location);
            scope.put("cookieName", OPENID_CONNECT_RETAINED_ANCHOR);
            scope.put("sameSite", sameSite);
            scope.put("useSecure", cookieSecureString);
            scope.put("root", request.getContextPath());

            Writer writer = response.getWriter();
            response.setContentType("text/html;charset=UTF-8");
            template.execute(scope, writer);
		}

        private boolean hasCookie(String cookieName) {
            if (request.getCookies() != null) {
                return !Arrays.stream(request.getCookies())
                        .filter(c -> c.getName().equals(cookieName))
                        .findFirst()
                        .map(Cookie::getValue)
                        .orElse("").isEmpty();
            }
            return false;
        }
	}
}
