package com.xebialabs.xlrelease.auth.oidc.config;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.URI;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.crypto.spec.SecretKeySpec;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Profile;
import org.springframework.core.env.Environment;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.http.converter.FormHttpMessageConverter;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.oauth2.client.*;
import org.springframework.security.oauth2.client.endpoint.DefaultAuthorizationCodeTokenResponseClient;
import org.springframework.security.oauth2.client.endpoint.NimbusJwtClientAuthenticationParametersConverter;
import org.springframework.security.oauth2.client.endpoint.OAuth2AuthorizationCodeGrantRequestEntityConverter;
import org.springframework.security.oauth2.client.http.OAuth2ErrorResponseErrorHandler;
import org.springframework.security.oauth2.client.registration.ClientRegistration;
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository;
import org.springframework.security.oauth2.client.registration.ClientRegistrations;
import org.springframework.security.oauth2.client.registration.InMemoryClientRegistrationRepository;
import org.springframework.security.oauth2.client.web.*;
import org.springframework.security.oauth2.core.AuthorizationGrantType;
import org.springframework.security.oauth2.core.ClientAuthenticationMethod;
import org.springframework.security.oauth2.core.DelegatingOAuth2TokenValidator;
import org.springframework.security.oauth2.core.OAuth2TokenValidator;
import org.springframework.security.oauth2.core.http.converter.OAuth2AccessTokenResponseHttpMessageConverter;
import org.springframework.security.oauth2.jose.jws.JwsAlgorithm;
import org.springframework.security.oauth2.jose.jws.MacAlgorithm;
import org.springframework.security.oauth2.jose.jws.SignatureAlgorithm;
import org.springframework.security.oauth2.jwt.*;
import org.springframework.security.oauth2.server.resource.authentication.JwtAuthenticationConverter;
import org.springframework.security.web.authentication.logout.LogoutSuccessHandler;
import org.springframework.security.web.authentication.session.SessionAuthenticationStrategy;
import org.springframework.security.web.savedrequest.NullRequestCache;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;
import com.nimbusds.jose.Algorithm;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.jwk.*;

import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.plumbing.authentication.RedirectToUrlSuccessHandler;
import com.xebialabs.deployit.security.UserService;
import com.xebialabs.platform.sso.oidc.authentication.CustomOidcIdTokenDecoderFactory;
import com.xebialabs.platform.sso.oidc.exceptions.UnsupportedOidcConfigurationException;
import com.xebialabs.platform.sso.oidc.policy.ClaimsToGrantedAuthoritiesPolicy;
import com.xebialabs.platform.sso.oidc.policy.impl.DefaultClaimsToGrantedAuthoritiesPolicy;
import com.xebialabs.platform.sso.oidc.policy.impl.GrantedAuthoritiesExtractor;
import com.xebialabs.platform.sso.oidc.service.XLOidcUserService;
import com.xebialabs.platform.sso.oidc.web.CustomAuthorizationRequestResolver;
import com.xebialabs.platform.sso.oidc.web.OidcLogoutSuccessHandler;
import com.xebialabs.xlrelease.auth.oidc.policy.impl.OidcUserProfileCreationPolicy;
import com.xebialabs.xlrelease.auth.oidc.web.BeforeLicenseCheckFilter;
import com.xebialabs.xlrelease.auth.oidc.web.OpenIdConnectRetainAnchorFilter;
import com.xebialabs.xlrelease.auth.oidc.web.XlReleaseLoginFormFilter;
import com.xebialabs.xlrelease.auth.oidc.web.authentication.Http401LoginUrlAuthenticationEntryPoint;
import com.xebialabs.xlrelease.auth.oidc.web.handlers.OidcLoginFailureHandler;
import com.xebialabs.xlrelease.auth.oidc.web.handlers.XlReleaseLoginFailureHandler;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.security.IdentityProvider;
import com.xebialabs.xlrelease.security.authentication.policy.UserProfileCreationPolicy;
import com.xebialabs.xlrelease.server.jetty.SSLConstants;
import com.xebialabs.xlrelease.service.UserProfileService;
import com.xebialabs.xlrelease.spring.configuration.XlrProfiles;

import static com.xebialabs.deployit.booter.local.utils.Strings.isBlank;
import static com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.xlrelease.auth.oidc.web.XlReleaseLoginFormFilter.LOGIN_PATH_NAME;
import static org.slf4j.LoggerFactory.getLogger;


@Configuration
@Profile(XlrProfiles.OIDC_AUTH)
public class OpenIdConnectConfig implements EnvironmentAware {
    public static final String OIDC_LOGIN_PATH_NAME = "oauth2/authorization/xl-release";

    public static final String OIDC_LOGIN = "/".concat(OIDC_LOGIN_PATH_NAME);

    public static final String OIDC_PROCESSING_URL = "/oidc-login";

    public static final String INTERNAL_USER_LOGIN_SUCCESS_URL = "#/login?reloadUserDetails";

    private static final Logger logger = getLogger(OpenIdConnectConfig.class);

    private static final String REGISTRATION_ID = "xl-release";

    private static final String JWSALG_CONFIG_PATH = "xl.security.auth.providers.oidc.clientAuthJwt.jwsAlg";

    private static final Map<MacAlgorithm, String> MAC_ALGORITHM_MAPPING = Map.of(
            MacAlgorithm.HS256, "HmacSHA256",
            MacAlgorithm.HS384, "HmacSHA384",
            MacAlgorithm.HS512, "HmacSHA512"
    );

    private static final Map<JwsAlgorithm, Curve> ESA_ALGORITHM_CURVE_MAPPING = Map.of(
            SignatureAlgorithm.ES256, Curve.P_256,
            SignatureAlgorithm.ES384, Curve.P_384,
            SignatureAlgorithm.ES512, Curve.P_521
    );

    private static final Map<JwsAlgorithm, JWSAlgorithm> ESA_ALGORITHM_MAP = Map.of(
            SignatureAlgorithm.ES256, JWSAlgorithm.ES256,
            SignatureAlgorithm.ES384, JWSAlgorithm.ES384,
            SignatureAlgorithm.ES512, JWSAlgorithm.ES512
    );

    private static final Map<JwsAlgorithm, JWSAlgorithm> RSA_ALGORITHM_MAP = Map.of(
            SignatureAlgorithm.RS256, JWSAlgorithm.RS256,
            SignatureAlgorithm.RS384, JWSAlgorithm.RS384,
            SignatureAlgorithm.RS512, JWSAlgorithm.RS512,
            SignatureAlgorithm.PS256, JWSAlgorithm.PS256,
            SignatureAlgorithm.PS384, JWSAlgorithm.PS384,
            SignatureAlgorithm.PS512, JWSAlgorithm.PS512);

    public static final String CHECK_JWS_ALG_MESSAGE = "Failed to find a Signature Verifier. Ensure you have" +
            " configured a valid JWS Algorithm.";

    private JWK jwk;

    private String clientId;

    private String clientSecret;

    private String clientAuthMethod;

    private String clientAuthJWSAlg;

    private String tokenKeyId;

    private String keyStorePath;

    private String keyStorePassword;

    private String keyStoreType;

    private String keyAlias;

    private String keyPassword;

    private String issuer;

    private String keyRetrievalUri;

    private String accessTokenUri;

    private String userAuthorizationUri;

    private String logoutUri;

    private String redirectUri;

    private String postLogoutRedirectUri;

    private String rolesClaim;

    private String fullNameClaim;

    private String userNameClaim;

    private String emailClaim;

    private String externalIdClaim;

    private String scopes;

    private String idTokenJWSAlg;

    private String accessTokenIssuer;

    private String accessTokenAudience;

    private String accessTokenKeyUri;

    private String accessTokenJWSAlg;

    private String accessTokenSecretKey;

    private String proxyHost;

    private Integer proxyPort;

    @Override
    public void setEnvironment(final Environment environment) {
        this.clientId = environment.getProperty("xl.security.auth.providers.oidc.clientId");
        this.clientSecret = environment.getProperty("xl.security.auth.providers.oidc.clientSecret", "");
        this.clientAuthMethod = environment.getProperty("xl.security.auth.providers.oidc.clientAuthMethod", "");
        this.clientAuthJWSAlg = environment.getProperty(JWSALG_CONFIG_PATH, "");
        this.tokenKeyId = environment.getProperty("xl.security.auth.providers.oidc.clientAuthJwt.tokenKeyId", "");
        this.keyStorePath = environment.getProperty("xl.security.auth.providers.oidc.clientAuthJwt.keyStore.path", "");
        this.keyStorePassword = environment.getProperty("xl.security.auth.providers.oidc.clientAuthJwt.keyStore.password", "");
        this.keyStoreType = environment.getProperty("xl.security.auth.providers.oidc.clientAuthJwt.keyStore.type", "");
        this.keyAlias = environment.getProperty("xl.security.auth.providers.oidc.clientAuthJwt.key.alias", "");
        this.keyPassword = environment.getProperty("xl.security.auth.providers.oidc.clientAuthJwt.key.password", "");
        this.issuer = environment.getProperty("xl.security.auth.providers.oidc.issuer");
        this.keyRetrievalUri = environment.getProperty("xl.security.auth.providers.oidc.keyRetrievalUri", "");
        this.accessTokenUri = environment.getProperty("xl.security.auth.providers.oidc.accessTokenUri", "");
        this.userAuthorizationUri = environment.getProperty("xl.security.auth.providers.oidc.userAuthorizationUri", "");
        this.logoutUri = environment.getProperty("xl.security.auth.providers.oidc.logoutUri", "");
        this.redirectUri = environment.getProperty("xl.security.auth.providers.oidc.redirectUri", "");
        this.postLogoutRedirectUri = environment.getProperty("xl.security.auth.providers.oidc.postLogoutRedirectUri", "");
        this.rolesClaim = environment.getProperty("xl.security.auth.providers.oidc.rolesClaim");
        this.fullNameClaim = environment.getProperty("xl.security.auth.providers.oidc.fullNameClaim");
        this.userNameClaim = environment.getProperty("xl.security.auth.providers.oidc.userNameClaim");
        this.emailClaim = environment.getProperty("xl.security.auth.providers.oidc.emailClaim");
        this.externalIdClaim = environment.getProperty("xl.security.auth.providers.oidc.externalIdClaim", "");
        this.scopes = environment.getProperty("xl.security.auth.providers.oidc.scopes");
        this.idTokenJWSAlg = environment.getProperty("xl.security.auth.providers.oidc.idTokenJWSAlg", "");
        this.accessTokenIssuer = environment.getProperty("xl.security.auth.providers.oidc.access-token.issuer", "");
        this.accessTokenAudience = environment.getProperty("xl.security.auth.providers.oidc.access-token.audience", "");
        this.accessTokenKeyUri = environment.getProperty("xl.security.auth.providers.oidc.access-token.keyRetrievalUri", "");
        this.accessTokenJWSAlg = environment.getProperty("xl.security.auth.providers.oidc.access-token.jwsAlg", "");
        this.accessTokenSecretKey = environment.getProperty("xl.security.auth.providers.oidc.access-token.secretKey", "");
        this.proxyHost = environment.getProperty("xl.security.auth.providers.oidc.proxyHost", "");
        this.proxyPort = environment.getProperty("xl.security.auth.providers.oidc.proxyPort", Integer.class);
    }

    // register the login provider in the application context for the AuthResource
    @Bean
    public IdentityProvider identityProvider() {
        return new IdentityProvider("oidc", String.format(".%s", OIDC_LOGIN), "./logout");
    }

    @Bean
    @Autowired
    public InMemoryClientRegistrationRepository clientRegistrationRepository(ServerConfiguration serverConfiguration) {
        return new InMemoryClientRegistrationRepository(this.clientRegistration(serverConfiguration.getServerUrl()));
    }

    @Bean
    @Autowired
    public UserProfileCreationPolicy userProfileCreationPolicy(UserProfileService userProfileService, UserService userService) {
        return new OidcUserProfileCreationPolicy(userProfileService, userService, emailClaim, fullNameClaim, externalIdClaim);
    }

    @Bean
    public ClaimsToGrantedAuthoritiesPolicy claimsToGrantedAuthoritiesPolicy() {
        return new DefaultClaimsToGrantedAuthoritiesPolicy(rolesClaim);
    }

    @Bean
    @Autowired
    public OAuth2AuthorizationRequestResolver customAuthorizationRequestResolver(XlrConfig xlrConfig, ClientRegistrationRepository clientRegistrationRepository) {
        Map<String, Object> additionalParameters = xlrConfig.oidcAdditionalParameters();

        return new CustomAuthorizationRequestResolver(clientRegistrationRepository,
                OAuth2AuthorizationRequestRedirectFilter.DEFAULT_AUTHORIZATION_REQUEST_BASE_URI, additionalParameters);
    }

    @Bean
    @Autowired
    public OAuth2AuthorizedClientService authorizedClientService(ClientRegistrationRepository clientRegistrationRepository) {
        return new InMemoryOAuth2AuthorizedClientService(clientRegistrationRepository);
    }

    @Bean
    @Autowired
    public XLOidcUserService xlrOidcUserService(ClaimsToGrantedAuthoritiesPolicy claimsToGrantedAuthoritiesPolicy) {
        return new XLOidcUserService(claimsToGrantedAuthoritiesPolicy);
    }

    @Bean
    @Autowired
    public OAuth2AuthorizedClientRepository authorizedClientRepository(OAuth2AuthorizedClientService authorizedClientService) {
        return new AuthenticatedPrincipalOAuth2AuthorizedClientRepository(authorizedClientService);
    }

    @Bean
    public Http401LoginUrlAuthenticationEntryPoint loginUrlAuthenticationEntryPoint() {
        ServerConfiguration serverConfig = ServerConfiguration.getInstance();
        String derivedPath = serverConfig.getDerivedServerUrl();
        String derivedUrl = derivedPath.endsWith("/") ? derivedPath : derivedPath + "/";
        if (!serverConfig.getServerUrl().equals(derivedUrl)) {
            return new Http401LoginUrlAuthenticationEntryPoint(serverConfig.getServerUrl() + OIDC_LOGIN_PATH_NAME);
        }
        return new Http401LoginUrlAuthenticationEntryPoint(OIDC_LOGIN);
    }

    @Bean
    public JwtDecoderFactory<ClientRegistration> idTokenDecoderFactory() {
        RestTemplate restTemplate = new RestTemplate();
        restTemplate.setRequestFactory(getClientHttpRequestFactory());

        JwsAlgorithm jwsAlgorithm = getJwsAlgorithm(idTokenJWSAlg, "idTokenJWSAlg",
                JwsAlgorithmTypeSupported.MAC_AND_SIGNATURE_ALGORITHM_SUPPORTED);

        CustomOidcIdTokenDecoderFactory idTokenDecoderFactory = new CustomOidcIdTokenDecoderFactory();
        idTokenDecoderFactory.setJwsAlgorithmResolver(clientRegistration -> jwsAlgorithm);
        idTokenDecoderFactory.setRestOperations(restTemplate);
        return idTokenDecoderFactory;
    }

    @Bean
    @Autowired
    public JwtDecoder jwtDecoder(ServerConfiguration serverConfiguration) {
        JwsAlgorithm jwsAlgorithm = getJwsAlgorithm(accessTokenJWSAlg, "jwsAlg",
                JwsAlgorithmTypeSupported.MAC_AND_SIGNATURE_ALGORITHM_SUPPORTED);

        NimbusJwtDecoder jwtDecoder;
        if (SignatureAlgorithm.class.isAssignableFrom(jwsAlgorithm.getClass())) {
            String jwkSetUri = this.clientRegistration(serverConfiguration.getServerUrl()).getProviderDetails().getJwkSetUri();
            if (isNotBlank(accessTokenKeyUri)) {
                jwkSetUri = accessTokenKeyUri;
            }

            RestTemplate restTemplate = new RestTemplate();
            restTemplate.setRequestFactory(getClientHttpRequestFactory());

            jwtDecoder = NimbusJwtDecoder
                    .withJwkSetUri(jwkSetUri)
                    .jwsAlgorithm((SignatureAlgorithm) jwsAlgorithm)
                    .restOperations(restTemplate)
                    .build();
        } else if (MacAlgorithm.class.isAssignableFrom(jwsAlgorithm.getClass())) {
            checkArgument(isNotBlank(accessTokenSecretKey), getPlaceholderErrorMessage("xl.security.auth.providers.oidc.access-token.secretKey"));

            SecretKeySpec secretKeySpec = new SecretKeySpec(accessTokenSecretKey.getBytes(StandardCharsets.UTF_8),
                    MAC_ALGORITHM_MAPPING.get(jwsAlgorithm));
            jwtDecoder = NimbusJwtDecoder
                    .withSecretKey(secretKeySpec)
                    .macAlgorithm((MacAlgorithm) jwsAlgorithm)
                    .build();
        } else {
            throw new UnsupportedOidcConfigurationException("Failed to find a Signature Verifier. Ensure you have configured a valid JWS Algorithm.");
        }

        OAuth2TokenValidator<Jwt> jwtValidator = getJwtOAuth2TokenValidator();
        jwtDecoder.setJwtValidator(jwtValidator);
        return jwtDecoder;
    }

    @Bean
    public JwtAuthenticationConverter jwtAuthenticationConverter() {
        JwtAuthenticationConverter jwtAuthenticationConverter = new JwtAuthenticationConverter();
        jwtAuthenticationConverter.setPrincipalClaimName(userNameClaim);
        jwtAuthenticationConverter.setJwtGrantedAuthoritiesConverter(new GrantedAuthoritiesExtractor(claimsToGrantedAuthoritiesPolicy()));
        return jwtAuthenticationConverter;
    }

    @Bean
    @Autowired
    public DefaultAuthorizationCodeTokenResponseClient authorizationCodeTokenResponseClient(ClientRegistrationRepository clientRegistrationRepository) {
        RestTemplate restTemplate = new RestTemplate(Arrays.asList(
                new FormHttpMessageConverter(), new OAuth2AccessTokenResponseHttpMessageConverter()));
        restTemplate.setRequestFactory(getClientHttpRequestFactory());
        restTemplate.setErrorHandler(new OAuth2ErrorResponseErrorHandler());

        DefaultAuthorizationCodeTokenResponseClient authorizationCodeTokenResponseClient = new DefaultAuthorizationCodeTokenResponseClient();
        authorizationCodeTokenResponseClient.setRestOperations(restTemplate);

        ClientRegistration clientRegistration = clientRegistrationRepository.findByRegistrationId(REGISTRATION_ID);
        if (clientRegistration.getClientAuthenticationMethod().equals(ClientAuthenticationMethod.CLIENT_SECRET_JWT) ||
                clientRegistration.getClientAuthenticationMethod().equals(ClientAuthenticationMethod.PRIVATE_KEY_JWT)) {
            setJwk(clientRegistration);
            OAuth2AuthorizationCodeGrantRequestEntityConverter requestEntityConverter =
                    new OAuth2AuthorizationCodeGrantRequestEntityConverter();
            requestEntityConverter.addParametersConverter(
                    new NimbusJwtClientAuthenticationParametersConverter<>(jwkResolver));
            authorizationCodeTokenResponseClient.setRequestEntityConverter(requestEntityConverter);
        }

        return authorizationCodeTokenResponseClient;
    }

    Function<ClientRegistration, JWK> jwkResolver = clientRegistration -> {
        if (clientRegistration.getClientAuthenticationMethod().equals(ClientAuthenticationMethod.CLIENT_SECRET_JWT)
                || clientRegistration.getClientAuthenticationMethod().equals(ClientAuthenticationMethod.PRIVATE_KEY_JWT)) {
            return jwk;
        }
        return null;
    };

    @Bean
    @Autowired
    public LogoutSuccessHandler xlrOidcLogoutSuccessHandler(ClientRegistrationRepository clientRegistrationRepository) {
        return new OidcLogoutSuccessHandler(clientRegistrationRepository, postLogoutRedirectUri, LOGIN_PATH_NAME);
    }

    @Bean
    public OidcLoginFailureHandler oidcLoginFailureHandler() {
        return new OidcLoginFailureHandler();
    }

    @Bean
    public XlReleaseLoginFailureHandler xlreleaseLoginFailureHandler() {
        return new XlReleaseLoginFailureHandler();
    }

    @Bean
    public NullRequestCache nullRequestCache() {
        return new NullRequestCache();
    }

    @Bean
    @Autowired
    public OpenIdConnectRetainAnchorFilter openIdConnectRetainAnchorFilter(XlrConfig xlrConfig, ServerConfiguration serverConfiguration) {
        OpenIdConnectRetainAnchorFilter openIdConnectRetainAnchorFilter = new OpenIdConnectRetainAnchorFilter(xlrConfig.server_http_cookie_sameSite().getAttributeValue());
        if (!serverConfiguration.isSsl()) {
            openIdConnectRetainAnchorFilter.setSecureCookie(serverConfiguration.isSecureCookieEnabled());
        }
        return openIdConnectRetainAnchorFilter;

    }

    @Bean
    public BeforeLicenseCheckFilter beforeLicenseCheckFilter() {
        return new BeforeLicenseCheckFilter();
    }

    @Bean
    @Autowired
    public XlReleaseLoginFormFilter xlReleaseLoginFormFilter(
            @Qualifier("authenticationManager") AuthenticationManager authenticationManager,
            SessionAuthenticationStrategy sessionAuthenticationStrategy
    ) {
        XlReleaseLoginFormFilter filter = new XlReleaseLoginFormFilter(defaultOidcMustacheTemplateSettings());
        filter.setAuthenticationManager(authenticationManager);
        filter.setSessionAuthenticationStrategy(sessionAuthenticationStrategy);
        filter.setAuthenticationSuccessHandler(new RedirectToUrlSuccessHandler(INTERNAL_USER_LOGIN_SUCCESS_URL));
        filter.setAuthenticationFailureHandler(xlreleaseLoginFailureHandler());
        return filter;
    }

    @Bean
    @Autowired
    public OAuth2AuthorizedClientManager authorizedClientManager(ClientRegistrationRepository clientRegistrationRepository, OAuth2AuthorizedClientRepository authorizedClientRepository) {
        OAuth2AuthorizedClientProvider authorizedClientProvider =
                OAuth2AuthorizedClientProviderBuilder.builder()
                        .authorizationCode()
                        .refreshToken()
                        .clientCredentials()
                        .build();

        DefaultOAuth2AuthorizedClientManager authorizedClientManager =
                new DefaultOAuth2AuthorizedClientManager(
                        clientRegistrationRepository, authorizedClientRepository);
        authorizedClientManager.setAuthorizedClientProvider(authorizedClientProvider);

        return authorizedClientManager;
    }

    public Map<String, String> defaultOidcMustacheTemplateSettings() {
        Map<String, String> scope = new HashMap<>();
        scope.put("productLogo", "static/0/styles/img/digital-ai-release.svg");
        scope.put("productColor", "#498500");
        scope.put("productIcon", "static/0/styles/img/favicon.ico");
        scope.put("productName", "Digital.ai Release");
        return scope;
    }

    private ClientRegistration clientRegistration(String serverUrl) {
        /*
            Try to create client registration with below two options:

            1. Try to fetch required configuration uris via issuer discovery uri i.e. http://server.com/.well-known/openid-configuration
            see, https://openid.net/specs/openid-connect-discovery-1_0.html#WellKnownRegistry
            User can still override required uris by providing appropriate placeholder configuration as before.


            2. If discovery uri is not present, try to create required configuration by resolving below placeholders. Validation has to be present for this.
                xl.security.auth.providers.oidc.userAuthorizationUri
                xl.security.auth.providers.oidc.accessTokenUri
                xl.security.auth.providers.oidc.keyRetrievalUri
                xl.security.auth.providers.oidc.logoutUri
         */

        ClientRegistration.Builder clientRegistrationBuilder = getClientRegistrationBuilder();
        ClientAuthenticationMethod authenticationMethod = getClientAuthMethod();

        clientRegistrationBuilder

                .clientId(clientId)
                .clientAuthenticationMethod(authenticationMethod)
                .authorizationGrantType(AuthorizationGrantType.AUTHORIZATION_CODE)
                .userInfoUri("")
                .scope(Stream.of(scopes.split(",")).collect(Collectors.toList()))
                .userNameAttributeName(userNameClaim);

        // clientSecret is required if authentication method is not none and not private key jwt
        if (!authenticationMethod.equals(ClientAuthenticationMethod.NONE) &&
                !authenticationMethod.equals(ClientAuthenticationMethod.PRIVATE_KEY_JWT)) {
            checkArgument(isNotBlank(clientSecret), getPlaceholderErrorMessage("xl.security.auth.providers.oidc.clientSecret"));
            clientRegistrationBuilder.clientSecret(clientSecret);
        }

        // validate fields required for authentication method private_key_jwt
        if (authenticationMethod.equals(ClientAuthenticationMethod.PRIVATE_KEY_JWT)) {
            if (isBlank(keyStorePath)) {
                keyStorePath = System.getProperty(SSLConstants.KEYSTORE_PROPERTY());
                keyStorePassword = System.getProperty(SSLConstants.KEYSTORE_PASSWORD_PROPERTY());
            }
            String clientAuthJwtProperty = "xl.security.auth.providers.oidc.clientAuthJwt";
            checkArgument(isNotBlank(keyStorePath), getPlaceholderErrorMessage(clientAuthJwtProperty + ".keyStore.path"));
            checkArgument(isNotBlank(keyAlias), getPlaceholderErrorMessage(clientAuthJwtProperty + ".key.alias"));
        }

        // get redirectUri from configuration. If null, construct it with serverUrl
        if (isNotBlank(redirectUri)) {
            clientRegistrationBuilder.redirectUri(redirectUri);
        } else {
            clientRegistrationBuilder.redirectUri(URI.create(serverUrl + OIDC_PROCESSING_URL).normalize().toString());
        }

        if (isNotBlank(userAuthorizationUri)) {
            clientRegistrationBuilder.authorizationUri(getEncodedUriString(userAuthorizationUri));
        }
        if (isNotBlank(accessTokenUri)) {
            clientRegistrationBuilder.tokenUri(getDecodedUriString(accessTokenUri));
        }
        if (isNotBlank(keyRetrievalUri)) {
            clientRegistrationBuilder.jwkSetUri(getEncodedUriString(keyRetrievalUri));
        }

        ClientRegistration clientRegistration = clientRegistrationBuilder.build();

        Map<String, Object> configurationMetadata = clientRegistration.getProviderDetails().getConfigurationMetadata();
        Map<String, Object> updatedConfigurationMetadata = new HashMap<>(configurationMetadata);

        if (isNotBlank(logoutUri)) {
            updatedConfigurationMetadata.put("end_session_endpoint", getDecodedUriString(logoutUri));
        }

        updatedConfigurationMetadata.put("fullNameClaim", fullNameClaim);
        updatedConfigurationMetadata.put("emailClaim", emailClaim);
        updatedConfigurationMetadata.put("rolesClaim", rolesClaim);

        return ClientRegistration.withClientRegistration(clientRegistration)
                .providerConfigurationMetadata(updatedConfigurationMetadata)
                .build();
    }

    private ClientRegistration.Builder getClientRegistrationBuilder() {
        String decodedIssuer = getDecodedUriString(issuer);
        try {
            if (decodedIssuer.matches("\\S+")) {
                return ClientRegistrations.fromOidcIssuerLocation(decodedIssuer).registrationId(REGISTRATION_ID);
            } else {
                throw new UnsupportedOidcConfigurationException(
                        String.format("Whitespace characters in issuer url [%s] is not supported. Recommendation is to avoid using spaces in URLs, and instead use hyphens to separate words.",
                                decodedIssuer));
            }
        } catch (IllegalArgumentException re) {
            logger.warn(String.format("Unable to resolve Configuration with the provided Issuer of [%s]", issuer));
            validateOidcConfiguration();
            return ClientRegistration.withRegistrationId(REGISTRATION_ID);
        } catch (UnsupportedOidcConfigurationException e) {
            logger.warn(e.getMessage());
            validateOidcConfiguration();
            return ClientRegistration.withRegistrationId(REGISTRATION_ID);
        }
    }

    private ClientHttpRequestFactory getClientHttpRequestFactory() {
        SimpleClientHttpRequestFactory requestFactory = new SimpleClientHttpRequestFactory();

        if (isNotBlank(proxyHost)) {
            Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxyHost, proxyPort));
            requestFactory.setProxy(proxy);
        }
        return requestFactory;
    }

    private JwsAlgorithm getJwsAlgorithm(String jwsAlgorithm, String propertyName,
                                         JwsAlgorithmTypeSupported supported) {
        Map<String, JwsAlgorithm> jwsAlgorithms = computeJwsAlgorithmMap(supported);
        if (isNotBlank(jwsAlgorithm)) {
            if (jwsAlgorithms.containsKey(jwsAlgorithm)) {
                return jwsAlgorithms.get(jwsAlgorithm);
            } else {
                throw new IllegalArgumentException(String.format("%s value [%s] is not supported. Ensure you have configured a valid JWS Algorithm.", propertyName, jwsAlgorithm));
            }
        } else {
            switch (supported) {
                case MAC_ALGORITHM_SUPPORTED:
                    return MacAlgorithm.HS256;
                case SIGNATURE_ALGORITHM_SUPPORTED:
                case MAC_AND_SIGNATURE_ALGORITHM_SUPPORTED:
                default:
                    return SignatureAlgorithm.RS256;
            }
        }

    }

    private Map<String, JwsAlgorithm> computeJwsAlgorithmMap(JwsAlgorithmTypeSupported typeSupported) {
        Map<String, JwsAlgorithm> jwsAlgorithms = new HashMap<>();
        if (typeSupported == JwsAlgorithmTypeSupported.SIGNATURE_ALGORITHM_SUPPORTED ||
                typeSupported == JwsAlgorithmTypeSupported.MAC_AND_SIGNATURE_ALGORITHM_SUPPORTED) {
            jwsAlgorithms.putAll(EnumSet.allOf(SignatureAlgorithm.class).stream().collect(Collectors.toMap(SignatureAlgorithm::getName, x -> x)));
        }
        if (typeSupported == JwsAlgorithmTypeSupported.MAC_ALGORITHM_SUPPORTED ||
                typeSupported == JwsAlgorithmTypeSupported.MAC_AND_SIGNATURE_ALGORITHM_SUPPORTED) {
            jwsAlgorithms.putAll(EnumSet.allOf(MacAlgorithm.class).stream().collect(Collectors.toMap(MacAlgorithm::getName, x -> x)));
        }
        return jwsAlgorithms;
    }

    private ClientAuthenticationMethod getClientAuthMethod() {
        Map<String, ClientAuthenticationMethod> clientAuthMethods = new HashMap<>();
        clientAuthMethods.put("client_secret_basic", ClientAuthenticationMethod.CLIENT_SECRET_BASIC);
        clientAuthMethods.put("client_secret_post", ClientAuthenticationMethod.CLIENT_SECRET_POST);
        clientAuthMethods.put("client_secret_jwt", ClientAuthenticationMethod.CLIENT_SECRET_JWT);
        clientAuthMethods.put("private_key_jwt", ClientAuthenticationMethod.PRIVATE_KEY_JWT);
        clientAuthMethods.put("none", ClientAuthenticationMethod.NONE);

        if (isNotBlank(clientAuthMethod)) {
            String authMethod = clientAuthMethod.toLowerCase();
            if (clientAuthMethods.containsKey(authMethod)) {
                return clientAuthMethods.get(authMethod);
            } else {
                throw new IllegalArgumentException(String.format("clientAuthMethod value [%s] is not supported. Ensure you have configured a valid client authentication method.", clientAuthMethod));
            }
        } else {
            return ClientAuthenticationMethod.CLIENT_SECRET_POST;
        }
    }

    private OAuth2TokenValidator<Jwt> getJwtOAuth2TokenValidator() {
        List<OAuth2TokenValidator<Jwt>> validators = new ArrayList<>();
        validators.add(new JwtTimestampValidator());
        validators.add(new JwtIssuerValidator(isNotBlank(accessTokenIssuer) ? accessTokenIssuer : issuer));
        validators.add(new JwtClaimValidator<List<String>>(JwtClaimNames.AUD,
                aud -> aud.contains(isNotBlank(accessTokenAudience) ? accessTokenAudience : clientId)));
        return new DelegatingOAuth2TokenValidator<>(validators);
    }

    /**
     * Encode and decode URI to prevent double encoding by spring-security
     */

    private String getEncodedUriString(String uri) {
        try {
            return URI.create(uri).toASCIIString();
        } catch (IllegalArgumentException e) {
            return UriComponentsBuilder.fromUriString(uri).build().toUri().toASCIIString();
        }
    }

    private String getDecodedUriString(String uri) {
        return URLDecoder.decode(uri, StandardCharsets.UTF_8);
    }

    private void validateOidcConfiguration() {
        checkArgument(isNotBlank(userAuthorizationUri), getPlaceholderErrorMessage("xl.security.auth.providers.oidc.userAuthorizationUri"));
        checkArgument(isNotBlank(accessTokenUri), getPlaceholderErrorMessage("xl.security.auth.providers.oidc.accessTokenUri"));
        checkArgument(isNotBlank(keyRetrievalUri), getPlaceholderErrorMessage("xl.security.auth.providers.oidc.keyRetrievalUri"));
        checkArgument(isNotBlank(logoutUri), getPlaceholderErrorMessage("xl.security.auth.providers.oidc.logoutUri"));
    }

    private String getPlaceholderErrorMessage(String placeholder) {
        return String.format("Could not resolve placeholder '%s' in value ${%s}", placeholder, placeholder);
    }

    private void setJwk(ClientRegistration clientRegistration) {
        if (clientRegistration.getClientAuthenticationMethod().equals(ClientAuthenticationMethod.CLIENT_SECRET_JWT)) {
            JwsAlgorithm jwsAlgorithm = getJwsAlgorithm(clientAuthJWSAlg, JWSALG_CONFIG_PATH,
                    JwsAlgorithmTypeSupported.MAC_ALGORITHM_SUPPORTED);

            if (MacAlgorithm.class.isAssignableFrom(jwsAlgorithm.getClass())) {
                SecretKeySpec secretKey = new SecretKeySpec(
                        clientRegistration.getClientSecret().getBytes(StandardCharsets.UTF_8),
                        MAC_ALGORITHM_MAPPING.get(jwsAlgorithm));
                jwk = new OctetSequenceKey.Builder(secretKey)
                        .algorithm(new Algorithm(((MacAlgorithm) jwsAlgorithm).name()))
                        .keyUse(KeyUse.SIGNATURE)
                        .keyID(UUID.randomUUID().toString())
                        .build();
            } else {
                throw new UnsupportedOidcConfigurationException(CHECK_JWS_ALG_MESSAGE);
            }
        } else if (clientRegistration.getClientAuthenticationMethod().equals(ClientAuthenticationMethod.PRIVATE_KEY_JWT)) {
            JwsAlgorithm jwsAlgorithm = getJwsAlgorithm(clientAuthJWSAlg, JWSALG_CONFIG_PATH,
                    JwsAlgorithmTypeSupported.SIGNATURE_ALGORITHM_SUPPORTED);

            KeyPair keyPair = getKeyPair();

            if (SignatureAlgorithm.class.isAssignableFrom(jwsAlgorithm.getClass())) {
                if (RSA_ALGORITHM_MAP.containsKey(jwsAlgorithm)) {
                    jwk = getRSAKey(keyPair, RSA_ALGORITHM_MAP.get(jwsAlgorithm));
                } else if (ESA_ALGORITHM_MAP.containsKey(jwsAlgorithm)) {
                    jwk = getESAKey(keyPair, ESA_ALGORITHM_MAP.get(jwsAlgorithm),
                            ESA_ALGORITHM_CURVE_MAPPING.get(jwsAlgorithm));
                } else {
                    throw new UnsupportedOidcConfigurationException(CHECK_JWS_ALG_MESSAGE);
                }
            } else {
                throw new UnsupportedOidcConfigurationException(CHECK_JWS_ALG_MESSAGE);
            }
        }
    }

    private KeyPair getKeyPair() {
        KeyPair keyPair;
        File file = new File(keyStorePath);
        if (!file.exists()) {
            throw new IllegalArgumentException(String.format("Cannot find keystore [%s]", keyStorePath));
        }
        if (isBlank(keyStorePassword)) {
            logger.warn("The keystore [{}] is not protected by a password. It is recommended to secure it using a password.", keyStorePath);
        }

        try (FileInputStream in = new FileInputStream(file)) {
            KeyStore keystore = KeyStore.getInstance((isNotBlank(keyStoreType)) ? keyStoreType : KeyStore.getDefaultType());
            keystore.load(in, (isNotBlank(keyStorePassword)) ? keyStorePassword.toCharArray() : null);

            PrivateKey privateKey = (PrivateKey) keystore.getKey(keyAlias, (isNotBlank(keyPassword)) ? keyPassword.toCharArray() : null);
            Certificate cert = keystore.getCertificate(keyAlias);
            PublicKey publicKey = cert.getPublicKey();
            keyPair = new KeyPair(publicKey, privateKey);
        } catch (UnrecoverableKeyException | KeyStoreException | NoSuchAlgorithmException | IOException | CertificateException | RuntimeException e) {
            throw new DeployitException(e);
        }
        return keyPair;
    }

    private RSAKey getRSAKey(KeyPair keyPair, Algorithm algorithm) {
        RSAKey.Builder builder = new RSAKey.Builder((RSAPublicKey) keyPair.getPublic())
                .privateKey(keyPair.getPrivate())
                .algorithm(algorithm)
                .keyUse(KeyUse.SIGNATURE);
        if (isNotBlank(tokenKeyId)) {
            builder.keyID(tokenKeyId);
        } else {
            logger.debug("Building RSAKey without token k-id. Token key Id not provided in configuration.");
        }
        return builder.build();
    }

    private ECKey getESAKey(KeyPair keyPair, Algorithm algorithm, Curve curve) {
        ECKey.Builder builder = new ECKey.Builder(curve, (ECPublicKey) keyPair.getPublic())
                .privateKey((ECPrivateKey) keyPair.getPrivate())
                .algorithm(algorithm)
                .keyUse(KeyUse.SIGNATURE);
        if (isNotBlank(tokenKeyId)) {
            builder.keyID(tokenKeyId);
        } else {
            logger.debug("Building ECKey without token k-id. Token key Id not provided in configuration.");
        }
        return builder.build();
    }

    private enum JwsAlgorithmTypeSupported {
        MAC_ALGORITHM_SUPPORTED,
        SIGNATURE_ALGORITHM_SUPPORTED,
        MAC_AND_SIGNATURE_ALGORITHM_SUPPORTED
    }
}
