#
# Copyright 2017 XEBIALABS
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

import sys
import string
import time
import com.xhaus.jyson.JysonCodec as json
from xlrelease.HttpRequest import HttpRequest
from azure.util import error
from azure.AzureServer import AzureServer

class ReleaseClient(AzureServer):
    """Exceptions are documented in the same way as classes.

    The __init__ method may be documented in either the class level
    docstring, or as a docstring on the __init__ method itself.

    Either form is acceptable, but the two should not be mixed. Choose one
    convention to document the __init__ method and be consistent with it.
    """

    __json = 'application/json'

    def __init__(self, azure_server, project):
        if azure_server is None:
            error('No server provided.')

        azure_server['url'] = "https://vsrm.dev.azure.com/%s/%s/" % (azure_server['organization'], project)

        if str(azure_server['authenticationMethod']) == 'OAuth2':
            access_token_url = "https://login.microsoftonline.com/%s/oauth2/v2.0/token" % azure_server['tenantId']
            azure_server['accessTokenUrl'] = access_token_url
            azure_server['password'] = AzureServer.get_access_token(azure_server)
            azure_server['authenticationMethod'] = 'Bearer'
        self.request = HttpRequest(azure_server)

    def createRelease(self, releaseName, releaseDescription, buildNumber):
        releaseDefinitionId = self.getReleaseDefinitionId(releaseName)
        artifacts = self.getDefaultArtifacts(releaseDefinitionId, buildNumber, releaseDescription)

        url = '_apis/release/releases?api-version=7.1-preview.8'
        response = self.request.post(url, json.dumps(artifacts), contentType=self.__json)

        if response.status != 200:
            error("Error creating release.", response)

        release = json.loads(response.response)

        return release['id']

    def getReleaseDefinitionId(self, releaseName):
        if not releaseName:
            raise Exception("Release Name is a mandatory field.")

        url = '_apis/release/definitions?searchText=%s&api-version=7.1-preview.4' % releaseName
        response = self.request.get(url)

        if response.status == 401 or response.status == 403:
            error("Authentication failed", response)
        
        if response.status != 200:
            error("Incorrect Project or Release name", response)

        release = json.loads(response.response)

        if not release['value'] or not release['value'][0]:
            error("Incorrect Release Name '%s'" % releaseName, response)
            
        buildId = None
        for value in release['value']:
            if value['name'] == releaseName:
                buildId = value['id']
                
        if not buildId:
            error("Incorrect Release Name '%s'" % releaseName, response)
            
        return buildId

    def getDefaultArtifacts(self, releaseDefinitionId, buildNumber, releaseDescription = ""):
        url = '_apis/release/artifacts/versions?releaseDefinitionId=%s&api-version=7.1-preview.1' % releaseDefinitionId
        response = self.request.get(url)

        artifacts = json.loads(response.response)

        return ReleaseStartMetadata(artifacts, releaseDefinitionId, buildNumber, releaseDescription)

    def getReleaseStatus(self, releasePipelineId, pipelineStageName):
        if not pipelineStageName:
            error('PipelineStageName is a mandatory field. No Pipeline Stage Name supplied.')

        release = self.getReleaseById(releasePipelineId)
        status = None

        for environment in release['environments']:
            if environment['name'].upper() == pipelineStageName.upper():
                status = environment['status']

        if status is None:
            error("No stage with name '%s' found" % pipelineStageName)

        return status

    def getReleaseById(self, releasePipelineId):
        if not releasePipelineId:
            error('releasePipelineId is a mandatory field. No Release Pipeline Id supplied.')

        response = self.request.get('_apis/release/releases/%s?api-version=7.1-preview.8' % releasePipelineId)

        if response.status != 200:
            error("Release pipeline with id '%s' was not found." % releasePipelineId, response)

        return json.loads(response.response)

    def getReleaseEnvironmentId(self, releasePipelineId, pipelineStageName):
        if not pipelineStageName:
            error('PipelineStageName is a mandatory field. No Pipeline Stage Name supplied.')

        release = self.getReleaseById(releasePipelineId)
        id = None

        for environment in release['environments']:
            if environment['name'].upper() == pipelineStageName.upper():
                id = environment['id']

        if id is None:
            error("No stage with name '%s' found" % pipelineStageName)

        return id

    def startDeployment(self, releasePipelineId, pipelineStageName, comment):
        envId = self.getReleaseEnvironmentId(releasePipelineId, pipelineStageName)

        url = '_apis/release/releases/%s/environments/%s?api-version=7.1-preview.7' % (releasePipelineId, envId)
        body = '{"status": "inprogress","comment":"%s"}' % comment

        response = self.request.patch(url, body, contentType=self.__json)

        if response.status != 200:
            error("Failed to start the deployment for stage '%s'." % pipelineStageName, response)

    def getApprovals(self, releasePipelineId):
        url = '_apis/release/approvals?releasePipelineIdsFilter=%s&api-version=7.1-preview.3' % releasePipelineId

        response = self.request.get(url)

        if response.status != 200:
            error("Failed to retrieve the approvals for the release '%s'." % releasePipelineId)

        return json.loads(response.response)

    def approveDeployment(self, releasePipelineId, pipelineStageName, comment):
        if not releasePipelineId:
            error('releasePipelineId is a mandatory field. No Release Pipeline Id supplied.')

        if pipelineStageName is None:
            error('PipelineStageName is a mandatory field. No Pipeline Stage Name supplied.')

        attempts = 0

        while attempts < 3:
            approvals = self.getApprovals(releasePipelineId)

            if (approvals['count'] == 0):
                if (attempts == 2):
                    error("No approvals available for the release '%s'." % releasePipelineId)
                else:
                    time.sleep(5)
                    attempts += 1
            else:
                break

        approvalId = None

        for approval in approvals['value']:
            if approval['releaseEnvironment']['name'].upper() == pipelineStageName.upper():
                approvalId = approval['id']

        if approvalId is None:
            error("No approvals found for stage '%s'" % pipelineStageName)

        url = '_apis/release/approvals/%s?api-version=7.1-preview.3' % approvalId
        body = '{"status": "approved","comment":"%s"}' % comment

        response = self.request.patch(url, body, contentType=self.__json)

        if response.status != 200:
            error("Failed to approve the deployment for the environment '%s'." % pipelineStageName, response)

class ReleaseStartMetadata(object):
    def __init__(self, artifacts, releaseDefinitionId, buildNumber, releaseDescription = ""):

        self.definitionId = releaseDefinitionId
        self.description = releaseDescription
        self.buildNumber = buildNumber
        self.artifacts = []

        for item in artifacts['artifactVersions']:
            artifact = ArtifactMetadata(item, buildNumber)
            self.artifacts.append(artifact)

    def __json__(self):
        artifacts = []
        for a in self.artifacts:
            artifacts.append(a.__json__())

        artifacts_json = ",".join(artifacts).join(("[","]"))

        return u"""{"definitionId":%s, "description":"%s", "artifacts":%s}""" % (self.definitionId, self.description, artifacts_json)

class ArtifactMetadata(object):
    def __init__(self, artifacts, buildVersionInput):
        self.alias = artifacts['alias']
        version = None
        definition_name = None

        default_version = artifacts['defaultVersion']

        if artifacts['versions']:
            latest_version = artifacts['versions'][0]
            if default_version is None:
                default_version = latest_version;
            #definition name is returned only when artifact source is build pipeline and not repo
            if "definitionName" in latest_version:
                definition_name = latest_version["definitionName"]
        else:
            error("Build runs not available for '%s'." % self.alias)

        if buildVersionInput is None:
            version = default_version

        elif buildVersionInput and definition_name in buildVersionInput:
            build_version_str = buildVersionInput[definition_name]

            for a in artifacts['versions']:
                if a["name"] == build_version_str:
                    version = a
                    break
            if version is None:
                error("Incorrect build number for '%s'." % self.alias)

        elif buildVersionInput and definition_name not in buildVersionInput:
            version = default_version

        else:
            version = default_version

        self.instance_reference = BuildVersion(version)

    def __json__(self):
        return u"""{"alias":"%s", "instanceReference":%s }""" % (self.alias, self.instance_reference.__json__())

class BuildVersion(object):
    def __init__(self, version):
        try:
            self.id = version['id']
            self.name = version['name']
            self.source_branch = version['sourceBranch']
            self.is_repo_artifact = True if 'commitMessage' in version.keys() else False
            self.commit_message = version['commitMessage'] if 'commitMessage' in version.keys() else ""
            self.source_repository_id = version['sourceRepositoryId'] if 'sourceRepositoryId' in version.keys() else ""
            self.source_repository_type = version['sourceRepositoryType'] if 'sourceRepositoryType' in version.keys() else ""
            self.source_version = version['sourceVersion'] if 'sourceVersion' in version.keys() else ""
        except:
            error("Build version missing values")


    def __json__(self):
        return u"""{"id":"%s", "name":"%s", "sourceBranch":"%s", "sourceVersion":"%s", "sourceRepositoryId":"%s", "sourceRepositoryType":"%s"}""" % (self.id, self.name, self.source_branch, self.source_version, self.source_repository_id, self.source_repository_type)
