#
# Copyright (c) 2020. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of Digital.ai, Inc. and its affiliates, subsidiaries, and licensors.
#

import json, time
from xlrelease.HttpRequest import HttpRequest
from org.apache.http.client import ClientProtocolException
from com.xebialabs.overthere import CmdLine
from com.xebialabs.overthere.util import CapturingOverthereExecutionOutputHandler, OverthereUtils
from com.xebialabs.overthere.local import LocalConnection
from java.lang import String
import org.slf4j.LoggerFactory as LoggerFactory

class BitbucketCloudClient(object):
    def __init__(self, server, username, password):
        self.logger = LoggerFactory.getLogger("com.xebialabs.bitbucket-plugin")
        if username not in [None, ""] or password not in [None, ""]:
            self.http_request = HttpRequest(server, username, password)
        else:
            self.http_request = HttpRequest(server)

    @staticmethod
    def get_client(server, username, password):
        return BitbucketCloudClient(server, username, password)

    def parse_output(self, lines):
        result_output = ""
        for line in lines:
            result_output = "\n".join([result_output, line])
        return result_output

    def api_call(self, method, endpoint, **options):
        try:
            options["method"] = method.upper()
            options["context"] = endpoint
            response = self.http_request.doRequest(**options)
        except ClientProtocolException:
            raise Exception("URL is not valid")
        if not response.isSuccessful():
            raise Exception("HTTP response code %s (%s)" % (response.getStatus(), response.errorDump()))
        return response

    def bitbucketcloud_createpullrequest(self, variables):
        endpoint = "/2.0/repositories/%s/pullrequests" % str(variables["repo_full_name"])
        content = """{
            "title": "%s",
            "description": "%s",
            "source": {
                "branch": {
                    "name": "%s"
                }
            },
            "destination": {
                "branch": {
                    "name": "%s"
                }
            },
            "close_source_branch": %s
        }""" % (str(variables["title"]),
                str(variables["description"]),
                str(variables["source"]),
                str(variables["target"]),
                str(variables["closebranch"]).lower())
        self.logger.warn(  "Submitting Pull Request %s using endpoint %s" % (content, endpoint) )
        response = self.api_call("POST",endpoint, body = content, contentType="application/json")
        data = json.loads(response.getResponse())
        self.logger.warn("Pull Request created with ID %s " % data["id"])
        return {"output" : data, "prid" : data["id"]}

    def bitbucketcloud_mergepullrequest(self, variables):
        endpoint = "/2.0/repositories/%s/pullrequests/%s/merge" % (str(variables["repo_full_name"]), str(variables["prid"]))
        content = """{
            "message": "%s",
            "close_source_branch": %s
        }""" % (str(variables["message"]),
                str(variables["closebranch"]).lower())
        self.logger.warn( "Merging Pull Request %s using endpoint %s" % (content, endpoint) )
        response = self.api_call("POST",endpoint, body = content, contentType="application/json")
        data = json.loads(response.getResponse())
        self.logger.warn( "Pull Request %s merged sucessfully with STATE : %s" % ( data["id"], data["state"]) )
        return {"output" : data}

    def bitbucketcloud_downloadcode(self, variables):
        downloadURL = "%s/%s/get/%s.zip" % (variables["server"]["url"].replace("api.","www."), variables["repo_full_name"], variables["branch"] )
        connection = LocalConnection.getLocalConnection()

        capturedOutput = ""

        self.logger.warn( "Cleaning up download folder : %s" % variables["downloadPath"] )
        command = CmdLine()
        command.addArgument("rm")
        command.addArgument("-rf")
        command.addArgument(variables["downloadPath"] + "/*")
        output_handler = CapturingOverthereExecutionOutputHandler.capturingHandler()
        error_handler = CapturingOverthereExecutionOutputHandler.capturingHandler()
        exit_code = connection.execute(output_handler, error_handler, command)
        capturedOutput = self.parse_output(output_handler.getOutputLines()) + self.parse_output(error_handler.getOutputLines())

        self.logger.warn( " Now downloading code in download folder : %s" % variables["downloadPath"] )
        command = CmdLine()
        script = """
            cd %s
            wget --user %s --password %s  -O code.zip %s
            unzip code.zip
            rm -rf *.zip
            foldername=`ls -d */`
            mv -f $foldername* `pwd`
            rm -rf $foldername
        """ % (variables["downloadPath"], self.http_request.username, self.http_request.password,  downloadURL )
        script_file = connection.getFile(OverthereUtils.constructPath(connection.getFile(variables["downloadPath"]), "extract.sh"))
        OverthereUtils.write(String(script).getBytes(), script_file)
        script_file.setExecutable(True)
        command.addArgument(script_file.getPath())
        output_handler = CapturingOverthereExecutionOutputHandler.capturingHandler()
        error_handler = CapturingOverthereExecutionOutputHandler.capturingHandler()
        exit_code = connection.execute(output_handler, error_handler, command)
        capturedOutput += self.parse_output(output_handler.getOutputLines()) + self.parse_output(error_handler.getOutputLines())

        command = CmdLine()
        command.addArgument("rm")
        command.addArgument("-f")
        command.addArgument(variables["downloadPath"] + "/extract.sh")
        output_handler = CapturingOverthereExecutionOutputHandler.capturingHandler()
        error_handler = CapturingOverthereExecutionOutputHandler.capturingHandler()
        exit_code = connection.execute(output_handler, error_handler, command)
        capturedOutput += self.parse_output(output_handler.getOutputLines()) + self.parse_output(error_handler.getOutputLines())

        return {"output": capturedOutput}

    def bitbucketcloud_commitsquery(self, variables):
        self.logger.warn("bitbucketcloud_commitsquery-> START")
        data = self.bitbucketcloud_querycommits(variables)
        commits = data
        #self.logger.warn( "Build commitList\n %s" % json.dumps(commits, indent=4, sort_keys=True) )
        commitList = []
        self.logger.warn("bitbucketcloud_commitsquery-> Loop over commits")
        for commit in commits:
            self.logger.warn( "message ~%s~" %  commit["message"] )
            commitList.append( commit["message"] )
        results = { "output": data, "commitList": commitList }
        self.logger.warn( "results\n %s" % json.dumps(results, indent=4, sort_keys=True) )
        return results

    def bitbucketcloud_querycommits(self, variables):
        endpoint_get = "/2.0/repositories/%s/commits/%s" % (variables["repo_full_name"], variables["branch"] )
        endpoint_get = "%s?limit=%s" % (endpoint_get, variables["results_limit"])
        if ( variables["tag"] is not None ):
            endpoint_get = "%s&at=refs/tags/%s" % (endpoint_get, variables["tag"])
        self.logger.warn( "endpoint = %s" % endpoint_get )
        response = self.api_call("GET", endpoint_get, contentType="application/json", Origin = variables["server"]["url"])
        data = response.getResponse()
        data = json.loads(data)["values"]
        self.logger.warn( "DATA2 = %s" %  json.dumps(data, indent=4, sort_keys=True) )
        return data

    def bitbucketcloud_querymergerequests(self, variables):
        endpoint = "2.0/repositories/%s/pullrequests?state=%s" % (variables["repo_full_name"], variables["state"])
        self.logger.warn( "URL = %s" % endpoint )
        response = self.api_call("GET", endpoint, contentType="application/json", Origin = variables["server"]["url"])
        data = json.loads( response.getResponse() )["values"]
        self.logger.warn( "merge_requests = %s" % json.dumps(data, indent=4, sort_keys=True) )
        return data
