#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries and licensors.
#
import json
import threading
from java.net import HttpCookie

from checkmarx.client.CheckmarxClient import CheckmarxClient


class CheckmarxClientUtil(object):
    __lock = threading.RLock()
    auth_cache = {}

    @staticmethod
    def create_checkmarx_client(container, csrf_token=None, cx_cookie=None):
        return CheckmarxClient.create_client(container, csrf_token, cx_cookie)

    @staticmethod
    def get_checkmarx_client(checkmarx_server, retry):
        params = CheckmarxClientUtil.get_configuration(checkmarx_server)
        cookies = CheckmarxClientUtil.get_cookie_data(params, retry)
        return CheckmarxClientUtil.create_checkmarx_client(params, cookies.get("CXCSRFToken"),
                                                           cookies.get("cxCookie"))

    @staticmethod
    def get(checkmarx_server, context, **options):
        checkmarx_client = CheckmarxClientUtil.get_checkmarx_client(checkmarx_server, retry=False)
        checkmarx_response = checkmarx_client.http_request.get(context, **options)
        if checkmarx_response.getStatus() == 401:
            checkmarx_client = CheckmarxClientUtil.get_checkmarx_client(checkmarx_server, retry=True)
            checkmarx_response = checkmarx_client.http_request.get(context, **options)
        return checkmarx_response

    @staticmethod
    def post(checkmarx_server, context, body, **options):
        checkmarx_client = CheckmarxClientUtil.get_checkmarx_client(checkmarx_server, retry=False)
        checkmarx_response = checkmarx_client.http_request.post(context, json.dumps(body), **options)
        if checkmarx_response.getStatus() == 401:
            checkmarx_client = CheckmarxClientUtil.get_checkmarx_client(checkmarx_server, retry=True)
            checkmarx_response = checkmarx_client.http_request.post(context, json.dumps(body), **options)
        return checkmarx_response

    @staticmethod
    def get_configuration(checkmarx_server):
        params = {'url': checkmarx_server.get('url'), 'username': checkmarx_server.get('username'),
                  'password': checkmarx_server.get('password'),
                  'proxyHost': checkmarx_server.get('proxyHost'), 'proxyPort': checkmarx_server.get('proxyPort'),
                  'title': checkmarx_server.get('title')}
        return params

    @staticmethod
    def get_cookie_data(params, retry):
        # Added Lock to modify cookie data at a time by one thread
        # Removing it may cause login failure due to multiple request on checkmarx server
        with CheckmarxClientUtil.__lock:
            cookies = {}
            # TODO:
            # Set ID to Configuration ID which will be available from 8.5
            # instead of title as it is not unique
            id = params['title']
            if id not in CheckmarxClientUtil.auth_cache:
                CheckmarxClientUtil.authenticate(id, params)
            elif retry \
                    or CheckmarxClientUtil.auth_cache[id]['cxCookie'] == "" \
                    or CheckmarxClientUtil.auth_cache[id]['CXCSRFToken'] == "":
                CheckmarxClientUtil.authenticate(id, params)

            cookies["cxCookie"] = CheckmarxClientUtil.auth_cache[id]['cxCookie']
            cookies["CXCSRFToken"] = CheckmarxClientUtil.auth_cache[id]['CXCSRFToken']
            return cookies

    @staticmethod
    def authenticate(id, params):
        checkmarx_client = CheckmarxClientUtil.create_checkmarx_client(params)
        response = checkmarx_client.login()
        response_headers = response.getHeaders()
        cookies = CheckmarxClientUtil.retrieve_cookies(response_headers)
        CheckmarxClientUtil.auth_cache[id] = {
            'cxCookie': cookies["cxCookie"],
            'CXCSRFToken': cookies["CXCSRFToken"]
        }

    @staticmethod
    def retrieve_cookies(headers):
        cookies = {}
        for header in headers:
            if "set-cookie" == header.getName().lower() or "set-cookie2" == header.getName().lower():
                set_cookies = HttpCookie.parse(str(header))
                for cookie in set_cookies:
                    cookies[cookie.getName()] = cookie.getValue()
        return cookies
