/*
 * Copyright (c) 2018. All rights reserved.
 *
 * This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries and licensors.
 */
'use strict';

(function () {

    var CxosaQueryTileViewController = function ($scope, CxosaQueryService, XlrTileHelper) {
        var vm = this;

        vm.tileConfigurationIsPopulated = tileConfigurationIsPopulated;

        function tileConfigurationIsPopulated() {
            return !_.isEmpty(vm.config.checkmarxServer);
        }

        $scope.filterByKey = function (item) {
            return (item.key !== 'Project Name'
                && item.key !== 'Scan Id'
                && item.key !== 'CxOSA Scan Date'
                && item.key !== 'Vulnerabilities Score'
                && item.key !== 'Total Libraries'
                && item.key !== 'Outdated Version Libraries');
        };

        function load(config) {
            if (tileConfigurationIsPopulated()) {
                vm.loading = true;
                CxosaQueryService.executeQuery(vm.tile.id, config).then(
                    function (response) {
                        var cxOsaData = response.data.data;
                        var lstData = [];
                        if (!cxOsaData.error) {
                            $scope.xlrTile.title = vm.tile.title;
                            lstData.push({
                                key: 'Scan Id',
                                value: cxOsaData['scanId']
                            });
                            lstData.push({
                                key: 'Project Name',
                                value: cxOsaData['projectName'],
                                url: cxOsaData['checkmarxUrl'] + '/CxWebClient/SPA/#/viewer/project/' + cxOsaData['projectId']
                            });
                            lstData.push({key: 'CxOSA Scan Date', value: cxOsaData['statisticsCalculationDate']});
                            lstData.push({key: 'Vulnerabilities Score', value: cxOsaData['vulnerabilityScore']});
                            lstData.push({key: 'Total Libraries', value: cxOsaData['totalLibraries']});
                            lstData.push({key: 'Outdated Version Libraries', value: cxOsaData['outdated']});
                            vm.result = {'lstData': lstData};

                            var categories = []
                            var counts = []
                            var colorList = []
                            var keys = []
                            if  (cxOsaData['dataChart'] == 'security') {
                                vm.showVulnerabilityData = true;
                                categories = ['Low', 'Medium', 'High'];
                                keys = ['lowVulnerabilityLibraries', 'mediumVulnerabilityLibraries', 'highVulnerabilityLibraries'];
                                counts = [];
                                keys.forEach(function (key) {
                                    counts.push(cxOsaData[key]);
                                });
                                colorList = ['#fcd866', '#fd8d10', '#d94c3d'];
                            } else {
                                vm.showVulnerabilityData = false;
                                categories = ['Unknown', 'No Risk', 'Medium', 'High'];
                                keys = ['unknownRisk', 'lowRisk', 'mediumRisk', 'highRisk'];
                                counts = [];
                                keys.forEach(function (key) {
                                    counts.push(cxOsaData[key]);
                                });
                                colorList = ['#bbbbbb', '#08b153', '#fd8d10', '#d94c3d'];
                            }
                            vm.riskReportBarChartData = createEchartOptions(categories, counts, colorList);
                        } else {
                            var error = ''
                            if (cxOsaData.error instanceof Array) {
                                for (var index = 0; index < cxOsaData.error.length; index++) {
                                    error = error + cxOsaData.error[index]['msg']
                                }
                            } else {
                                error = cxOsaData.error
                            }
                            vm.result = {'lstData': lstData, 'error': error};
                        }

                    }
                ).finally(function () {
                    vm.loading = false;
                });
            }
        }

        function createEchartOptions(categories, counts, colorList) {
            var DEFAULT_MAX = 30;
            return {
                grid: {
                    top:    '10%',
                    bottom: '10%',
                    left:   '15%',
                    right:  '10%'
                },
                yAxis: {
                    type: 'category',
                    data: categories,
                    axisLine: {show: false},
                    axisTick: {show: false},
                    axisPointer: {show: false}
                },
                xAxis: {
                    show: false,
                    type: 'value',
                    max: function () {
                        var valArray = counts.slice(0);
                        valArray.push(DEFAULT_MAX);
                        return Math.max(...valArray);
                    },
                    axisLabel: {
                        show: false
                    }
                },
                series: [{
                    data: counts,
                    type: 'bar',
                    itemStyle: {
                        normal: {
                            color: function (params) {
                                return colorList[params.dataIndex]
                            },
                            label: {
                                show: true,
                                position: 'right',
                                formatter: '{c}',
                                color: 'rgb(0,0,0)'
                            }
                        }
                    }
                }]
            };
        }


        function refresh() {
            load({params: {refresh: true}});
        }

        vm.$onInit = function () {
            if ($scope.xlrDashboard) {
                // summary page
                vm.release = $scope.xlrDashboard.release;
                vm.tile = $scope.xlrTile.tile;
                if (vm.tile.properties == null) {
                    vm.config = vm.tile.configurationProperties;
                } else {
                    // new style since 7.0
                    vm.config = vm.tile.properties;
                }
            }
            load();
        };

        vm.refresh = refresh;
    };

    CxosaQueryTileViewController.$inject = ['$scope', 'xlrelease.cxosa.CxosaQueryService', 'XlrTileHelper'];

    var CxosaQueryService = function (Backend) {

        function executeQuery(tileId, config) {
            return Backend.get("tiles/" + tileId + "/data", config);
        }

        return {
            executeQuery: executeQuery
        };
    };

    CxosaQueryService.$inject = ['Backend'];

    angular.module('xlrelease.cxosa.tile', []);
    angular.module('xlrelease.cxosa.tile').service('xlrelease.cxosa.CxosaQueryService', CxosaQueryService);
    angular.module('xlrelease.cxosa.tile').controller('cxosa.CxosaQueryTileViewController', CxosaQueryTileViewController);
    angular.module('xlrelease').directive("addScrollBar", function () {
        return {
            link: function (scope, element) {
                element.parent().css('overflowY', 'auto')
            }
        }
    });

})();

