package com.xebialabs.xlrelease.plugins.dashboard.cache

import java.util
import java.util.concurrent.TimeUnit

import com.google.common.cache.{CacheBuilder, CacheLoader}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.plugins.dashboard.domain.Tile
import grizzled.slf4j.Logging
import org.springframework.cache.annotation.{CachingConfigurerSupport, EnableCaching}
import org.springframework.cache.guava.GuavaCacheManager
import org.springframework.cache.interceptor.{CacheOperationInvocationContext, CacheResolver}
import org.springframework.cache.{Cache, CacheManager}
import org.springframework.context.annotation.{Bean, Configuration}

import scala.collection.JavaConversions._
import scala.util.{Failure, Success, Try}

object TileCacheConfiguration {
  final val TILE_CACHE = "tileCache"
  final val TILE_CACHE_MANAGER = "tileCacheManager"
  final val TILE_CACHE_KEY_GENERATOR = "tileCacheKeyGenerator"
  final val TILE_CACHE_RESOLVER = "tileCacheResolver"
}

@Configuration
@EnableCaching(proxyTargetClass = true)
class TileCacheConfiguration extends CachingConfigurerSupport with Logging {

  @Bean(name = Array(TileCacheConfiguration.TILE_CACHE_KEY_GENERATOR))
  override def keyGenerator(): TileCacheKeyGenerator = {
    new TileCacheKeyGenerator()
  }

  @Bean(name = Array(TileCacheConfiguration.TILE_CACHE_RESOLVER))
  override def cacheResolver: CacheResolver = {
    new CacheResolver {
      override def resolveCaches(context: CacheOperationInvocationContext[_]): util.Collection[_ <: Cache] = {
        val tileTypeName = context.getArgs.find(_.isInstanceOf[Tile]) match {
          case Some(tile) => tile.asInstanceOf[Tile].getType.toString
          case None => TileCacheConfiguration.TILE_CACHE
        }
        List(cacheManager().getCache(tileTypeName))
      }
    }
  }

  @Bean(name = Array(TileCacheConfiguration.TILE_CACHE_MANAGER))
  override def cacheManager: CacheManager = {
    val cacheManager: GuavaCacheManager = new GuavaCacheManager() {
      override def createNativeGuavaCache(name: String): com.google.common.cache.Cache[AnyRef, AnyRef] = {
        Try(Type.valueOf(name)) match {
          case Success(matchedTileType) =>
            // create a cache for this specific type
            cacheBuilderFrom(matchedTileType).build()
          case Failure(ex) =>
            logger.warn(s"Unable to find tile type with a name $name. Creating cache with default settings.")
            super.createNativeGuavaCache(name)
        }
      }

      override def setCacheLoader(cacheLoader: CacheLoader[AnyRef, AnyRef]) = {
        throw new UnsupportedOperationException("Cache loading not support by this manager implementation")
      }
    }
    val tileType = Type.valueOf(classOf[Tile])
    cacheManager.setCacheBuilder(cacheBuilderFrom(tileType))
    cacheManager
  }

  private def cacheBuilderFrom(tileType: Type): CacheBuilder[AnyRef, AnyRef] = {
    val expirationTime = tileType.getDescriptor.getPropertyDescriptor(Tile.PROPERTY_EXPIRATION_TIME).getDefaultValue.asInstanceOf[Integer].toLong
    val maxCacheEntries = tileType.getDescriptor.getPropertyDescriptor(Tile.PROPERTY_MAX_CACHE_ENTRIES).getDefaultValue.asInstanceOf[Integer].toLong
    val guavaTileCacheBuilder = CacheBuilder.newBuilder()
      .expireAfterAccess(expirationTime, TimeUnit.SECONDS)
      .maximumSize(maxCacheEntries)
      .softValues()
      .recordStats()
    guavaTileCacheBuilder
  }

}
