package com.xebialabs.xlrelease.plugins.dashboard.service

import javax.script.{ScriptContext, SimpleScriptContext}

import com.xebialabs.xlrelease.domain.{PythonScriptHelper, Release}
import com.xebialabs.xlrelease.plugins.dashboard.cache.TileCacheConfiguration
import com.xebialabs.xlrelease.plugins.dashboard.cache.TileCacheConfiguration._
import com.xebialabs.xlrelease.plugins.dashboard.domain.Tile
import com.xebialabs.xlrelease.script.ScriptService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.cache.annotation.{CacheConfig, CacheEvict, Cacheable}
import org.springframework.stereotype.Component

@Component
@CacheConfig(cacheResolver = TILE_CACHE_RESOLVER, keyGenerator = TILE_CACHE_KEY_GENERATOR, cacheNames = Array(TileCacheConfiguration.TILE_CACHE))
class TileScriptExecutor @Autowired() (scriptService: ScriptService) extends Logging {

  @CacheEvict(condition = "#resolvedTile.cacheEnabled")
  def evictFromCache(release: Release, resolvedTile: Tile, additionalVariables: Map[String, Any]) = {
    logger.debug(s"Executing refresh for a tile ${resolvedTile.getId}")
  }

  @Cacheable(condition = "#resolvedTile.cacheEnabled")
  def executeTileScript(release: Release, resolvedTile: Tile, additionalVariables: Map[String, Any]): AnyRef = {
    logger.debug(s"Executing tile script for a tile ${resolvedTile.getId}")
    val scriptContext = createScriptContext(release, resolvedTile, additionalVariables)

    val script = PythonScriptHelper.getScript(resolvedTile)

    scriptService.executeScriptWithWriter(script, scriptContext)
    scriptContext.getAttribute("data")
  }

  private def createScriptContext(release: Release, resolvedTile: Tile, additionalVariables: Map[String, Any]): ScriptContext = {
    val scriptContext = new SimpleScriptContext

    scriptService.addPropertiesToScriptContext(scriptContext, resolvedTile, resolvedTile.getConfigurationProperties)

    val properties: Map[String, Any] = additionalVariables ++ Map("release" -> release)
    properties.foreach { case (k, v) =>
      scriptContext.setAttribute(k, v, ScriptContext.ENGINE_SCOPE)
    }

    scriptService.addApiToContext(scriptContext)
    scriptService.addRepositoryServiceToContext(scriptContext)
    scriptContext
  }
}
