package com.xebialabs.xlrelease.plugins.dashboard.views

import java.util.{Map => JMap}

import com.google.common.base.Preconditions.checkArgument
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.xlrelease.domain.Task
import com.xebialabs.xlrelease.domain.Task.CATEGORY_INPUT
import com.xebialabs.xlrelease.plugins.dashboard.builder.DashboardBuilder.newTile
import com.xebialabs.xlrelease.plugins.dashboard.domain.Tile
import com.xebialabs.xlrelease.variable.VariableViewHelper.{fromView, toView}
import org.codehaus.jackson.annotate.JsonIgnoreProperties

import scala.beans.BeanProperty
import scala.collection.JavaConversions._
import scala.collection.JavaConverters._

object TileView {
  def apply(tile: Tile) = new TileView(tile)

  def toTile(tileView: TileView)(implicit repositoryService: RepositoryService) = {
    val tileDescriptor = Type.valueOf(tileView.`type`).getDescriptor
    val (propertiesWithValue, propertiesWithVariable) = tileView.getConfigurationProperties
      .map {
        case (name, viewValue) =>
          val pd = tileDescriptor.getPropertyDescriptor(name)
          checkArgument(pd != null, "Property [%s] is not part of synthetic type [%s]", name, tileView.`type`)
          name -> fromView(pd, viewValue)
      }
      .partition { case (name, (variable, value)) => variable == null }
    val properties: Map[String, AnyRef] = propertiesWithValue.mapValues(_._2).toMap
    val variableMapping: Map[String, String] = propertiesWithVariable.mapValues(_._1).toMap
    newTile
      .withType(tileView.`type`)
      .withId(tileView.getId)
      .withTitle(tileView.getTitle)
      .withRow(tileView.getRow)
      .withCol(tileView.getCol)
      .withWidth(tileView.getSizeX)
      .withProperties(properties)
      .withVariableMappings(variableMapping)
      .build
  }
}

@JsonIgnoreProperties(ignoreUnknown = true)
class TileView(val tile: Tile) {

  @BeanProperty
  var id: String = tile.getId
  @BeanProperty
  var `type`: String = tile.getType.toString

  @BeanProperty
  var uri: String = tile.getUri

  @BeanProperty
  var detailsUri: String = tile.getDetailsUri

  @BeanProperty
  var configurationUri: String = tile.getConfigurationUri

  @BeanProperty
  var title: String = tile.getTitle
  @BeanProperty
  var row: Integer = tile.getRow
  @BeanProperty
  var col: Integer = tile.getCol
  @BeanProperty
  var sizeX: Integer = tile.getWidth

  @BeanProperty
  var configurationProperties: JMap[String, AnyRef] = {
    tile.getType.getDescriptor.getPropertyDescriptors
      .filter(_.getCategory == CATEGORY_INPUT)
      .map(p => (p.getName, toView(p, tile.getVariableMapping.get(p.getName), p.get(tile))))
      .toMap.asJava
  }

  def this() {
    // Tile is virtual, but Jackson needs to create an empty view object
    this(new Tile())
  }
}
