package com.xebialabs.xlrelease.plugins.dashboard.api.internal

import java.util.{List => JList}
import javax.ws.rs._
import javax.ws.rs.core.{Context, MediaType, UriInfo}

import com.xebialabs.deployit.plugin.api.reflect.{Descriptor, DescriptorRegistry, Type}
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.xlrelease.plugins.dashboard.domain.Tile
import com.xebialabs.xlrelease.plugins.dashboard.repository.Dashboards
import com.xebialabs.xlrelease.plugins.dashboard.service.TileService
import com.xebialabs.xlrelease.plugins.dashboard.views.{TileData, TileView}
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.security.PermissionChecker
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

import scala.collection.JavaConversions._

@Path("/tiles")
@Controller
class TileResource @Autowired()(permissionChecker: PermissionChecker,
                                dashboards: Dashboards,
                                tilesService: TileService,
                                implicit val repositoryService: RepositoryService) {
  @GET
  @Produces(Array(MediaType.APPLICATION_JSON))
  @Path("/tile-types")
  def getTileTypes: JList[Descriptor] = {
    DescriptorRegistry.getSubtypes(Type.valueOf(classOf[Tile]))
      .map(_.getDescriptor)
      .filter(!_.isVirtual)
      .toList
      .sortBy(_.getLabel)
  }

  @GET
  @Path("/{tileId:.*}/data")
  @Produces(Array(MediaType.APPLICATION_JSON))
  def getTileData(@PathParam("tileId") tileId: String,
                  @Context info: UriInfo): TileData = {
    permissionChecker.checkView(releaseIdFrom(tileId))
    TileData(tilesService.executeTileDataScript(tileId, getQueryParameters(info)))
  }

  @GET
  @Path("{tileId:.*}")
  @Produces(Array(MediaType.APPLICATION_JSON))
  def getTile(@PathParam("tileId") tileId: String): TileView = {
    permissionChecker.checkEdit(releaseIdFrom(tileId))
    dashboards.findTileById(tileId)
  }

  @PUT
  @Path("{tileId:.*}")
  @Produces(Array(MediaType.APPLICATION_JSON))
  @Consumes(Array(MediaType.APPLICATION_JSON))
  def updateTile(@PathParam("tileId") tileId: String, tile: TileView): TileView = {
    permissionChecker.checkEdit(releaseIdFrom(tileId))
    tile.setId(tileId)
    import com.xebialabs.xlrelease.plugins.dashboard.views._
    dashboards.createOrUpdateTile(viewToTile(tile))
  }

  private[internal] def getQueryParameters(info: UriInfo): Map[String, Any] = info.getQueryParameters.map {
    case (k, v) => k -> info.getQueryParameters.getFirst(k)
  }.toMap

}
