package com.xebialabs.xlrelease.plugins.dashboard.repository

import java.util.{List => JList}

import com.google.common.base.Preconditions.checkNotNull
import com.xebialabs.deployit.plugin.api.reflect.{Descriptor, DescriptorRegistry, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.{ChangeSet, RepositoryService}
import com.xebialabs.xlrelease.plugins.dashboard.builder.DashboardBuilder.newDashboard
import com.xebialabs.xlrelease.plugins.dashboard.domain.{Dashboard, Tile}
import com.xebialabs.xlrelease.repository.Ids.{getName, releaseIdFrom}
import com.xebialabs.xlrelease.repository.{BulkRepositoryService, Ids, Releases}
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.variable.VariableHelper.fixUpVariableIds
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Repository

import scala.collection.JavaConversions._

@Repository
class Dashboards @Autowired()(repositoryService: RepositoryService, bulkRepositoryService: BulkRepositoryService, ciIdService: CiIdService, releases: Releases) extends Logging {

  private def findById(dashboardId: String): Dashboard = repositoryService read (dashboardId, false)

  def findByIdOrDefault(dashboardId: String): Dashboard =
    if (repositoryService.exists(dashboardId)) {
      findById(dashboardId)
    } else {
      val release = releases.findByIdIncludingArchived(releaseIdFrom(dashboardId))
      if (release.isArchived) {
        release.getProperty[JList[ConfigurationItem]]("extensions").find(ci => Ids.getName(dashboardId) == Ids.getName(ci.getId)) match {
          case None =>
            createDefaultDashboard(dashboardId)
          case Some(dashboard) =>
            dashboard.asInstanceOf[Dashboard]
        }
      } else {
        createDefaultDashboard(dashboardId)
      }
    }

  private def createDefaultDashboard(dashboardId: String): Dashboard = {
    newDashboard.withId(dashboardId).build.addDefaultTiles()
  }

  def findTileById(tileId: String): Tile = {
    val dashboardId = Ids.getParentId(tileId)
    val dashboard = findByIdOrDefault(dashboardId)
    dashboard.getTiles.find(_.getId == tileId).orNull
  }

  def createOrUpdateDashboard(dashboard: Dashboard): Dashboard = {
    checkNotNull(dashboard.getId)
    val changes: ChangeSet = new ChangeSet

    if (repositoryService.exists(dashboard.getId)) {
      // Update existing tiles
      changes createOrUpdate dashboard.getTiles

      // Delete removed tiles
      val existingTiles = repositoryService.read[Dashboard](dashboard.getId).getTiles
      val existingTileIds = existingTiles.map(_.getId).toList
      dashboard.updateTileIds()
      val newTileIds = dashboard.getTiles.map(_.getId).toList
      changes.delete(existingTileIds diff newTileIds)

    } else {
      changes.create(dashboard)
      dashboard.updateTileIds()
      changes.create(dashboard.getTiles)
    }

    repositoryService execute changes
    val savedDashboard = findById(dashboard.getId)

    scanAndAddVariables(savedDashboard.getId)

    savedDashboard
  }

  def createOrUpdateTile(tile: Tile): Tile = {
    val dashboardId = Ids.getParentId(tile.getId)
    if (!repositoryService.exists(dashboardId)) {
      createOrUpdateDashboard(findByIdOrDefault(dashboardId))
    }
    repositoryService.createOrUpdate(tile)

    scanAndAddVariables(dashboardId)

    repositoryService.read[Tile](tile.getId, false)
  }

  private def scanAndAddVariables(dashboardId: String): Unit = {
    val release = releases.findById(releaseIdFrom(dashboardId), false)
    val newVariables = release.scanAndAddNewVariables()
    fixUpVariableIds(release.getId, newVariables, ciIdService)
    bulkRepositoryService.create(newVariables)
  }

  implicit class DashboardExtensions(dashboard: Dashboard) {

    def updateTileIds(): Unit = {
      def nullId(t: Tile, id: String) = id == null || id.trim.isEmpty || id.trim.toLowerCase == "null" || id.trim.toLowerCase.endsWith(t.getType.toString.toLowerCase)
      dashboard.getTiles.foreach { t =>
        if (nullId(t, t.getId)) t.setId(dashboard.newTileId)
      }
    }

    def newTileId: String = {
      ciIdService.getUniqueId(Type.valueOf(classOf[Tile]), dashboard.getId)
    }

    def addTile(tile: Tile): Unit = {
      dashboard.setTiles(dashboard.getTiles :+ tile)
    }

    private def addTile(tileDescriptor: Descriptor): Unit = {
      val tempId = dashboard.getId + "/" + tileDescriptor.getType.toString
      addTile(tileDescriptor.newInstance[Tile](tempId))
    }

    def addDefaultTiles(): Dashboard = {
      for {
        tileDescriptor <- DescriptorRegistry.getSubtypes(Type.valueOf(classOf[Tile])).map(_.getDescriptor)
        if !tileDescriptor.isVirtual
        if tileDescriptor.shouldBeAddedTo(dashboard)
      } dashboard.addTile(tileDescriptor)
      dashboard
    }

    implicit class TileDescriptor(tileDescriptor: Descriptor) {
      def shouldBeAddedTo(dashboard: Dashboard): Boolean = {
        Option(tileDescriptor.getPropertyDescriptor("defaultDashboards"))
          .flatMap(default => Option(default.getDefaultValue.asInstanceOf[JList[String]]))
          .exists(_.contains(getName(dashboard.getId)))
      }
    }
  }
}