package com.xebialabs.xlrelease.runner.actors

import akka.actor.{ActorRef, Cancellable, PoisonPill}
import akka.cluster.sharding.{ClusterSharding, ShardRegion}
import akka.event.LoggingReceive
import akka.persistence._
import com.xebialabs.xlrelease.actors.XlrPersistentActor
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.runner.JobRunner
import com.xebialabs.xlrelease.runner.actors.JobRunnerActor._
import com.xebialabs.xlrelease.runner.domain.{JobId, _}
import com.xebialabs.xlrelease.runner.service.RunnerJobService
import com.xebialabs.xlrelease.scheduler.RunnerRegistry
import com.xebialabs.xlrelease.storage.domain.{LogEntry, LogEntryRef}
import com.xebialabs.xlrelease.storage.service.StorageService
import com.xebialabs.xlrelease.support.akka.spring.SpringActor
import org.apache.commons.io.IOUtils

import scala.concurrent.duration.DurationInt
import scala.concurrent.{ExecutionContextExecutor, Future}
import scala.util.{Failure, Success, Try, Using}

// In the cluster this actor should be a singleton and it should create required number of sharded job executors.
@SpringActor
class JobRunnerActor(xlrConfig: XlrConfig,
                     workerService: RunnerJobService,
                     storageService: StorageService,
                     jobExecutorActorFactory: JobExecutorActorFactory,
                     runnerRegistry: RunnerRegistry)
  extends XlrPersistentActor {

  var state: RunnerState = RunnerState()

  val snapShotInterval: Int = xlrConfig.xl.getInt("job-runner.snapshot-after")

  var lastWorkRequest: Cancellable = _

  private def selfRef: ActorRef = {
    if (xlrConfig.isClusterEnabled) {
      ClusterSharding(context.system).shardRegion(JobRunnerActor.SHARDING_TYPE_NAME)
    } else {
      self
    }
  }

  override def persistenceId: String = self.path.name

  override def receiveRecover: Receive = LoggingReceive {
    case event: RunnerEvent =>
      state = state.applyEvent(event)
    case SnapshotOffer(_, snapshot: RunnerState) =>
      state = snapshot
    case RecoveryCompleted =>
      log.info("Recovery completed")
      state = state.copy(requestInProgress = false)
      state.jobMap.foreachEntry { (jobId, _) =>
        jobExecutorActorFactory.resume(jobId)
      }
      state.lastEvent.foreach(handleEvent)
    case msg =>
      log.warning(s"Can't recover $msg")
  }

  override def receiveCommand: Receive = handleRunnerCommand orElse handleAkkaInternalMessage orElse handleUnknownMessage

  def handleRunnerCommand: Receive = LoggingReceive {
    case cmd: RunnerCommand =>
      handleCommand(cmd)
  }

  def handleAkkaInternalMessage: Receive = LoggingReceive {
    case SaveSnapshotSuccess(metadata) =>
      log.debug(s"Saved snapshot at sequence number ${metadata.sequenceNr}")
      val snapshotSelectionCriteria = SnapshotSelectionCriteria.create(metadata.sequenceNr - 1, Long.MaxValue)
      deleteMessages(metadata.sequenceNr)
      deleteSnapshots(snapshotSelectionCriteria)
    case SaveSnapshotFailure(metadata, cause) =>
      log.error(cause, s"Failed to save snapshot at sequence number ${metadata.sequenceNr}")
    case DeleteMessagesSuccess(toSequenceNr) =>
      log.debug(s"Deleted records from journal at sequence number $toSequenceNr")
    case DeleteMessagesFailure(cause, toSequenceNr) =>
      log.error(cause, s"Failed to delete records from journal at sequence number $toSequenceNr")
    case DeleteSnapshotsSuccess(criteria) =>
      log.debug(s"Deleted snapshot with criteria $criteria")
    case DeleteSnapshotsFailure(criteria, cause) =>
      log.error(cause, s"Failed to delete snapshot with criteria $criteria")
    case PoisonPill =>
      context.stop(self)
  }

  def handleUnknownMessage: Receive = {
    case msg =>
      log.warning(s"Dropping message $msg")
  }

  //noinspection ScalaStyle
  private def handleCommand(command: JobRunnerActor.RunnerCommand): Unit = command match {
    case StartRunner(newRunner) =>
      val event = RunnerStarted(newRunner)
      persist(event)(handleEvent)
    case StopRunner(_) =>
      // stop all executors?
      if (!state.isStopping) {
        val event = RunnerStopRequested()
        persist(event)(handleEvent)
      }
    case AskForWork(runnerId) =>
      log.debug(s"Current state: $state")
      if (state.isRunning) {
        // there can be only 1 request against the service at the time (e.g. service can return response after 50 seconds)
        // TODO currently we presume reliable delivery of messages, but it may happen that some message might be lost
        //  1. if the lost message is ProcessJob or ProcessAskForWorkFailure
        //     we will keep our state with "request in progress" and we will not be able to unblock
        //     in this scenario we should keep not "request in progress flag" but "instant of the last request"
        if (state.hasFreeCapacity && state.requestNotInProgress) {
          val event = AskedForWork()
          persist(event)(handleEvent)
        } else {
          // our restart mechanism - if anything goes wrong we will try again
          implicit val ec: ExecutionContextExecutor = scala.concurrent.ExecutionContext.global
          if (null != lastWorkRequest) {
            lastWorkRequest.cancel()
          }
          // TODO what if we get 5 very fast requests one after the other will they cancel each other? what if it's a tight loop?
          lastWorkRequest = context.system.scheduler.scheduleOnce(60.seconds, selfRef, AskForWork(runnerId))
        }
      }
    case ProcessAskForWorkFailure(runnerId) =>
      val event = AskForWorkDone()
      persist(event)(handleEvent)
    case ProcessJob(runnerId, jobData) =>
      val event = AskForWorkDone()
      persist(event)(handleEvent)
      jobData match {
        case data: ContainerJobData =>
          val event = ContainerJobReceived(data)
          persist(event)(handleEvent)
        case SomeJobData(jobId) =>
          log.warning("We don't know how to process this job.")
          workerService.failJob(jobId)
        case FailJobData(jobId) =>
          log.warning(s"We received request to fail a job $jobId. We should receive it as a command. Ignoring it.")
        case NoJobData() =>
        case StopWorkerJobData() =>
          log.warning("We received request to stop job. We should receive it as a command. Ignoring it.")
      }
    case FinishJob(_, jobResult) =>
      val event = JobFinished(jobResult)
      persist(event)(handleEvent)
    case AbortJob(_, jobId) =>
      if (state.jobMap.contains(jobId)) {
        log.debug(s"Requesting to abort job [$jobId]")
        val event = AbortJobRequested(jobId)
        persist(event)(handleEvent)
      } else {
        log.error(s"Job [$jobId] is not associated with this runner. Ignoring it.")
      }
    case SendLogEntry(_, logEntry) =>
      persist(LogEntryAdded(logEntry))(handleEvent)
    case SendDirectives(_, jobDirectives) =>
      persist(DirectivesAdded(jobDirectives))(handleEvent)
  }

  //noinspection ScalaStyle
  private def handleEvent(event: RunnerEvent): Unit = {
    state = state.applyEvent(event)
    takeSnapshot()
    event match {
      case RunnerStarted(runner) =>
        log.info(s"Runner $runner started")
        state.runner match {
          case Some(r) =>
            // register runner
            runnerRegistry.registerJobRunner(r)
            askForWork()
          case None =>
            log.error("There is no Runner. Stopping.")
            stop()
        }
      case RunnerStopRequested() =>
        log.info(s"Runner was requested to stop")
        state.runner.foreach(runner => runnerRegistry.unregisterJobRunner(runner))
      case AskedForWork() =>
        implicit val ec: ExecutionContextExecutor = scala.concurrent.ExecutionContext.global
        state.runner.foreach(r => Future(workerService.getJob(r.getId)).andThen {
          case Failure(exception) =>
            log.error(exception, "Error while fetching next job. Will ask again.")
            selfRef ! ProcessAskForWorkFailure(r.getId)
          case Success(jobData) =>
            selfRef ! ProcessJob(r.getId, jobData)
        })
      case AskForWorkDone() =>
        askForWork()
      case ContainerJobReceived(jobData) =>
        state.runner.foreach(runner => jobExecutorActorFactory.start(runner, jobData))
      case JobFinished(jobResult) =>
        // TODO if this is remote API call - what if remote call fails here
        //  if it's a persistent actor all events should be replayed and this one will be retried
        //  can this potentially end up in an endless loop?
        Try(workerService.finishJob(jobResult))
        askForWork()
      case AbortJobRequested(jobId) =>
        jobExecutorActorFactory.abort(jobId)
        askForWork()
      case DirectivesAdded(jobDirectives) =>
        workerService.executeDirectives(jobDirectives)
        askForWork()
      case LogEntryAdded(logEntryRef) =>
        Try {
          // parse log entry, see if there is a directive there and act accordingly?
          // runner and executor MUST have access to the same storage
          val payload = Using.resource(storageService.get(logEntryRef)) { content =>
            IOUtils.toByteArray(content)
          }
          val logEntry = LogEntry(
            taskId = logEntryRef.taskId,
            jobId = logEntryRef.jobId,
            chunk = logEntryRef.chunk,
            lastEntryTimestamp = logEntryRef.lastEntryTimestamp,
            payload = payload,
            uriScheme = "no-uri"
          )
          workerService.log(logEntry)
        }.recover {
          case t => log.error(t, s"Unable to log entry $logEntryRef")
        }
        askForWork()
    }
  }

  private def stop(): Unit = {
    if (null != lastWorkRequest) {
      lastWorkRequest.cancel()
    }
    if (xlrConfig.isClusterEnabled) {
      context.parent ! ShardRegion.Passivate(PoisonPill)
    } else {
      context.stop(self)
    }
  }

  private def askForWork(): Unit = {
    state.runner.foreach(r =>
      if (r.isEnabled && state.isRunning && state.hasFreeCapacity) {
        selfRef ! AskForWork(r.getId)
      }
    )
  }

  private def takeSnapshot(): Unit = {
    if (lastSequenceNr % snapShotInterval == 0 && lastSequenceNr != 0) {
      saveSnapshot(state)
    }
  }

}

object JobRunnerActor {

  final val SHARDING_TYPE_NAME = "job-runner"

  def actorName(runnerId: RunnerId): String = {
    s"job-runner-${runnerId.shortId()}"
  }

  type WorkerId = String

  case class WorkerMapping(workerId: WorkerId, workerActorRef: ActorRef)

  sealed trait RunnerStatus

  case object Initial extends RunnerStatus

  case object Running extends RunnerStatus

  case object Stopping extends RunnerStatus

  case object Stopped extends RunnerStatus

  case class RunnerState(status: RunnerStatus = Initial,
                         runner: Option[JobRunner] = None,
                         jobMap: Map[JobId, JobData] = Map.empty,
                         requestInProgress: Boolean = false,
                         lastEvent: Option[RunnerEvent] = None) {

    override def toString: WorkerId = {
      s"RunnerState(status = $status, requestInProgress = $requestInProgress, runner = $runner, lastEvent = $lastEvent, jobIds = ${jobMap.keys})"
    }

    def applyEvent(event: RunnerEvent): RunnerState = {
      val newState = event match {
        case RunnerStarted(runner) => copy(status = Running, runner = Some(runner))
        case RunnerStopRequested() => copy(status = Stopping, runner = None, requestInProgress = false)
        case AskedForWork() => copy(requestInProgress = true)
        case AskForWorkDone() => copy(requestInProgress = false)
        case ContainerJobReceived(jobData) => copy(jobMap = jobMap + (jobData.jobId -> jobData))
        case JobFinished(jobResult) => copy(jobMap = jobMap - jobResult.jobId)
        case AbortJobRequested(jobId) => copy(jobMap = jobMap - jobId)
        case LogEntryAdded(_) => this
        case DirectivesAdded(_) => this
      }
      /*
        Store last event in the state to use it during recovery. System deletes the events from journal after a snapshot has been saved.
        If there are no events in the database during recovery, lastEvent will be null and system won't able to handle last event. Due to this, it is required to save the event in state.
       */
      newState.copy(lastEvent = Some(event))
    }

    def isRunning: Boolean = status == Running

    def isStopping: Boolean = status == Stopping

    def hasFreeCapacity: Boolean = runner match {
      case Some(r) => jobMap.size < r.capacity
      case None => false
    }

    def requestNotInProgress: Boolean = !requestInProgress
  }

  sealed trait RunnerCommand {
    def runnerId: RunnerId
  }

  case class StartRunner(newRunner: JobRunner) extends RunnerCommand {
    override def runnerId: RunnerId = newRunner.getId
  }

  case class StopRunner(runnerId: RunnerId) extends RunnerCommand

  case class AskForWork(runnerId: RunnerId) extends RunnerCommand

  case class ProcessJob(runnerId: RunnerId, jobData: JobData) extends RunnerCommand

  case class ProcessAskForWorkFailure(runnerId: RunnerId) extends RunnerCommand

  case class FinishJob(runnerId: RunnerId, jobResult: JobResult) extends RunnerCommand

  case class AbortJob(runnerId: RunnerId, jobId: JobId) extends RunnerCommand

  case class SendLogEntry(runnerId: RunnerId, logEntryRef: LogEntryRef) extends RunnerCommand

  case class SendDirectives(runnerId: RunnerId, directives: Seq[JobDirective]) extends RunnerCommand

  sealed trait RunnerEvent

  case class RunnerStopRequested() extends RunnerEvent

  case class RunnerStarted(runner: JobRunner) extends RunnerEvent

  case class AskedForWork() extends RunnerEvent

  case class AskForWorkDone() extends RunnerEvent

  case class ContainerJobReceived(jobData: ContainerJobData) extends RunnerEvent

  case class JobFinished(jobResult: JobResult) extends RunnerEvent

  case class AbortJobRequested(jobId: JobId) extends RunnerEvent

  case class LogEntryAdded(logEntry: LogEntryRef) extends RunnerEvent

  case class DirectivesAdded(jobDirectives: Seq[JobDirective]) extends RunnerEvent

}
