package com.xebialabs.xlrelease.runner.docker.actors

import akka.actor.{ActorRef, PoisonPill}
import akka.cluster.sharding.{ClusterSharding, ShardRegion}
import akka.event.LoggingReceive
import akka.persistence.{DeleteMessagesFailure, DeleteMessagesSuccess, RecoveryCompleted}
import com.github.dockerjava.api.DockerClient
import com.xebialabs.xlrelease.actors.XlrPersistentActor
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.runner.actors.JobRunnerActor.{FinishJob, RunnerCommand}
import com.xebialabs.xlrelease.runner.actors.{JobRunnerActor, JobRunnerActorFactory}
import com.xebialabs.xlrelease.runner.docker.DockerClientFactory
import com.xebialabs.xlrelease.runner.docker.actors.DirectiveParser.Directive
import com.xebialabs.xlrelease.runner.docker.actors.DockerJobExecutorActor._
import com.xebialabs.xlrelease.runner.docker.domain.DockerOptions
import com.xebialabs.xlrelease.runner.domain.{JobId, _}
import com.xebialabs.xlrelease.storage.domain.{LogEntry, LogEntryRef}
import com.xebialabs.xlrelease.storage.service.StorageService
import com.xebialabs.xlrelease.support.akka.SerializableMsg
import com.xebialabs.xlrelease.support.akka.spring.SpringActor
import grizzled.slf4j.Logging
import org.springframework.util.StringUtils.hasText

import java.net.URI
import java.time.Instant
import java.time.format.DateTimeFormatter
import scala.concurrent.duration.DurationInt

@SpringActor
class DockerJobExecutorActor(val xlrConfig: XlrConfig,
                             jobRunnerActorFactory: JobRunnerActorFactory,
                             val storageService: StorageService)
  extends XlrPersistentActor with DockerService with LogEntryParser {

  private var dockerConnectionRetryCount: Int = 0
  private var state: DockerJobExecutorState = DockerJobExecutorState()
  private var lastRecoverableDockerJobEvent: DockerJobEvent = _

  private lazy val _directiveParser: DirectiveParser = DirectiveParser()

  override def persistenceId: String = self.path.name

  override def parser: DirectiveParser = _directiveParser

  def dockerClient: DockerClient = DockerClientFactory.apply(state.dockerOptions)

  override def receiveRecover: Receive = {
    case evt: DockerJobEvent =>
      state = state.applyEvent(evt)
      evt match {
        case event: RecoverableEvent => lastRecoverableDockerJobEvent = event
        case _: UnRecoverableEvent => ()
      }
    case RecoveryCompleted =>
      log.debug(s"[$persistenceId] Recovery completed")
      if (null != lastRecoverableDockerJobEvent) handleEvent(lastRecoverableDockerJobEvent)
    case msg =>
      log.warning(s"[$persistenceId] Can't recover $msg")
  }

  override def receiveCommand: Receive = LoggingReceive {
    case cmd: DockerJobCommand => handleCommand(cmd)
    case PoisonPill => context.stop(self)
    case msg => log.warning(s"[$persistenceId] Received unknown message $msg")
  }

  // scalastyle:off cyclomatic.complexity
  private def handleCommand(command: DockerJobCommand): Unit = command match {
    case ResumeJob(_) =>
      log.debug(s"Received resume job")
    case TerminateJob(_) =>
      // TODO what if we cannot connect to docker?
      persist(TerminateJobStarted)(handleEvent)
    case startJob: StartJob =>
      withCheckStatus(startJob, Initial)(persist(JobStarted(startJob.dockerOptions, startJob.jobData))(handleEvent))
    case pingResult: DockerPingResult =>
      val maxDockerRetryCount = state.jobData.maxRetryAttempts
      val dockerRetryDelay = state.jobData.retryDelay.seconds

      if (pingResult.success) {
        dockerConnectionRetryCount = 0
        state.dockerOperationRecoveryEvent.foreach(handleEvent)
      } else {
        dockerConnectionRetryCount += 1
        // just use non-state variable as retry counter and reset it once ping succeeds
        // schedule again ping in a while (do not call it immediately because it blocks)
        if (dockerConnectionRetryCount < maxDockerRetryCount) {
          context.system.scheduler.scheduleOnce(dockerRetryDelay) {
            ping(jobId)
          }(scala.concurrent.ExecutionContext.global)
        } else {
          // what if we leave task in FAILING state ? can we retry such task or fail it?
          // so if it is FAILING and we RETRY -> then we run this job stuff again from current state
          // but... if it is FAILING and we ABORT -> then we abort this task
          // ideally it should PAUSE task until someone decides to abort it or retry it
          handleEvent(DockerConnectRetryFailed)
        }
      }
    case operation: DockerOperation =>
      handleDockerOperationCommand(operation)
    case result: DockerOperationResult =>
      val event = result.event
      persist(event)(handleEvent)
    case AddLogEntry(logEntry, storedEntryUri) =>
      val logEntryRef = LogEntryRef.from(logEntry, storedEntryUri)
      val logEntryAddedEvent = LogEntryAdded(logEntryRef)
      val directives = parse(logEntry)
      val events: Seq[DockerJobEvent] = Seq(logEntryAddedEvent) ++ (if (directives.nonEmpty) {
        Seq(DirectivesAddedEvent(directives))
      } else {
        Seq()
      })
      persistAll(events)(handleEvent)
    // another set of events for directives / parsing?
    case StopExecutor(_) =>
      //TODO: deleteMessages will delete messages but it will still keep last record
      // we should schedule a job which will remove all deleted records from the database
      // frequency for the job can be once a day
      // see: https://github.com/akka/akka-persistence-jdbc/blob/v5.0.4/core/src/main/resources/reference.conf#L26
      deleteMessages(lastSequenceNr)
      context.become(stopExecutor)
  }

  private def handleDockerOperationCommand(operation: DockerOperation): Unit = operation match {
    case PullImage(_) =>
      withCheckStatus(operation, StartingJob)(persist(PullImageStarted)(handleEvent))
    case CreateContainer(_) =>
      withCheckStatus(operation, PullingImage)(persist(CreateContainerStarted)(handleEvent))
    case CreateContainerInputContext(_) =>
      withCheckStatus(operation, CreatingContainer)(persist(CreateContainerInputContextStarted)(handleEvent))
    case StartContainer(_) =>
      withCheckStatus(operation, CreatingContainerInputContext)(persist(StartContainerStarted)(handleEvent))
    case RetrieveLogs(_) =>
      withCheckStatus(operation, StartingContainer)(persist(LogRetrievalStarted)(handleEvent))
    case WaitForContainer(_) =>
      withCheckStatus(operation, RetrievingLogs)(persist(WaitForContainerStarted)(handleEvent))
    case CreateContainerOutputContext(_) =>
      withCheckStatus(operation, RunningContainer, TerminatingJob)(persist(CreateContainerOutputContextStarted)(handleEvent))
    case RemoveContainer(_) =>
      persist(RemoveContainerStarted)(handleEvent)
  }

  //noinspection ScalaStyle
  private def handleEvent(event: DockerJobEvent): Unit = {
    state = state.applyEvent(event)
    event match {
      case JobStarted(_, _) =>
        selfRef ! PullImage(jobId)
      case LogEntryAdded(logEntryRef) =>
        val runnerId = state.dockerOptions.runnerId
        sendMsgToRunner(JobRunnerActor.SendLogEntry(runnerId = runnerId, logEntryRef))
      case DirectivesAddedEvent(directives) =>
        val runnerId = state.dockerOptions.runnerId
        val taskId = state.jobData.taskId
        val jobDirectives = directives.map(d => JobDirective(jobId, taskId, directiveName = d.name, payload = d.payload))
        sendMsgToRunner(JobRunnerActor.SendDirectives(runnerId, jobDirectives))
      case started: DockerJobOperationStarted => started match {
        case PullImageStarted =>
          val taskId = state.jobData.taskId
          pullImage(taskId, jobId, state.jobData.taskImg, state.lastLogChunk)
        case CreateContainerStarted =>
          createContainer(jobId, state.jobData.taskImg, persistenceId)
        case CreateContainerInputContextStarted =>
          createContainerInputContext(jobId, state.containerId, state.jobData.inputContext.data)
        case StartContainerStarted =>
          startContainer(jobId, state.containerId)
        case LogRetrievalStarted =>
          captureContainerLog(state.jobData.taskId, jobId, state.containerId, state.lastLogChunk, state.lastLogTimestamp)
        case WaitForContainerStarted =>
          waitForContainer(jobId, state.containerId)
        case CreateContainerOutputContextStarted =>
          createContainerOutputContext(jobId, state.containerId)
        case RemoveContainerStarted =>
          removeContainer(jobId, state.containerId, persistenceId)
        case TerminateJobStarted =>
          terminateContainer(jobId, state.containerId, persistenceId, state.jobData.abortTimeout)
      }
      case completed: DockerJobOperationCompleted => completed match {
        case PullImageCompleted =>
          selfRef ! CreateContainer(jobId)
        case CreateContainerCompleted(_) =>
          selfRef ! CreateContainerInputContext(jobId)
        case CreateContainerInputContextCompleted =>
          selfRef ! StartContainer(jobId)
        case StartContainerCompleted =>
          selfRef ! RetrieveLogs(jobId)
        case LogRetrievalCompleted =>
          selfRef ! WaitForContainer(jobId)
        case WaitForContainerCompleted(_) =>
          selfRef ! CreateContainerOutputContext(jobId)
        case CreateContainerOutputContextCompleted(outputResult) =>
          val taskId = state.jobData.taskId
          val runnerId = state.dockerOptions.runnerId
          val exitStatusCode = state.containerStatusCode
          val jobResult = TaskJobSuccess(taskId, jobId, runnerId, exitStatusCode, PlainContextData(outputResult))
          sendMsgToRunner(FinishJob(runnerId, jobResult))
          selfRef ! RemoveContainer(jobId)
        case RemoveContainerCompleted =>
          selfRef ! StopExecutor(jobId)
        case TerminateJobCompleted =>
          selfRef ! CreateContainerOutputContext(jobId)
      }
      case failed: DockerJobOperationFailed => failed match {
        case ConnectionFailed(_) =>
          ping(jobId)
        case OperationFailed(_) => state.status match {
          case DockerJobExecutorActor.TerminatingJob =>
            selfRef ! CreateContainerOutputContext(jobId)
          case DockerJobExecutorActor.Stopping =>
            selfRef ! StopExecutor(jobId) //TODO: Do we want to fail the job as well?
          case _ =>
            failJob()
            if (hasText(state.containerId)) {
              selfRef ! RemoveContainer(jobId)
            } else {
              selfRef ! StopExecutor(jobId)
            }
        }
        case DockerConnectRetryFailed => state.dockerOperationRecoveryEvent match {
          case Some(value) => value match {
            case RemoveContainerStarted =>
              selfRef ! StopExecutor(jobId)
            case _ =>
              failJob()
              selfRef ! StopExecutor(jobId)
          }
          case None =>
            selfRef ! StopExecutor(jobId)
        }
      }
    }
  }

  private def failJob(): Unit = {
    val jobData = state.jobData
    val taskId = jobData.taskId
    val runnerId = state.dockerOptions.runnerId
    sendMsgToRunner(FinishJob(runnerId, TaskJobFailure(taskId, jobId, runnerId, state.outputResultOrExceptionTrace)))
  }

  private def jobId: JobId = state.jobData.jobId

  private def sendMsgToRunner(msg: RunnerCommand): Unit = {
    val jobRunner = jobRunnerActorFactory.create()
    jobRunner ! msg
  }

  private def withCheckStatus(cmd: DockerJobCommand, statuses: ExecutorStatus*)(fn: => Unit): Unit = {
    if (statuses.contains(state.status)) {
      fn
    } else {
      log.error(s"Current state is ${state.status}. Can't handle $cmd")
    }
  }

  def stopExecutor: Receive = LoggingReceive {
    case DeleteMessagesSuccess(_) =>
      log.debug(s"[$persistenceId] Successfully deleted all records from journal")
      stop()
    case DeleteMessagesFailure(cause, _) =>
      log.error(cause, s"[$persistenceId] Unable to delete records from journal")
      stop()
    case PoisonPill => context.stop(self)
    case msg =>
      log.warning(s"[$persistenceId] Received unknown message $msg")
  }

  override def selfRef: ActorRef = {
    if (xlrConfig.isClusterEnabled) {
      ClusterSharding(context.system).shardRegion(DockerJobExecutorActor.SHARDING_TYPE_NAME)
    } else {
      self
    }
  }

  private def stop(): Unit = {
    if (xlrConfig.isClusterEnabled) {
      context.parent ! ShardRegion.Passivate(PoisonPill)
    } else {
      context.stop(self)
    }
  }

}

// scalastyle:off number.of.methods
object DockerJobExecutorActor {

  final val SHARDING_TYPE_NAME: String = "docker-job-executor"

  final val INITIAL_DATE: String = DateTimeFormatter.ISO_INSTANT.format(Instant.ofEpochMilli(0)) // seconds since January 1, 1970 (midnight UTC/GMT)

  def actorName(jobId: JobId): String = s"job-executor-$jobId"

  // state
  case class DockerJobExecutorState(status: ExecutorStatus = Initial,
                                    containerId: String = null,
                                    outputResultOrExceptionTrace: String = null,
                                    dockerOptions: DockerOptions = null,
                                    containerStatusCode: Int = -1,
                                    jobData: ContainerJobData = null,
                                    lastLogChunk: Long = 0,
                                    lastLogTimestamp: String = INITIAL_DATE,
                                    dockerOperationRecoveryEvent: Option[DockerJobOperationStarted] = None
                                   ) extends Logging {

    //noinspection ScalaStyle
    def applyEvent(event: DockerJobEvent): DockerJobExecutorState = {
      val newState = event match {
        case JobStarted(dockerOptions, jobData) => copy(status = StartingJob, dockerOptions = dockerOptions, jobData = jobData)
        case DockerConnectRetryFailed =>
          val containerMsg = if (null != containerId) {
            s" Please check container [$containerId] and execute required operation manually."
          } else {
            ""
          }
          val exceptionMsg = if (null != outputResultOrExceptionTrace) {
            s"$outputResultOrExceptionTrace${System.lineSeparator()}$containerMsg"
          } else {
            s"System is unable to connect to docker server.$containerMsg"
          }
          copy(outputResultOrExceptionTrace = exceptionMsg)
        case PullImageStarted => copy(status = PullingImage)
        case PullImageCompleted => this
        case CreateContainerStarted => copy(status = CreatingContainer)
        case CreateContainerCompleted(containerId) => copy(containerId = containerId)
        case CreateContainerInputContextStarted => copy(status = CreatingContainerInputContext)
        case CreateContainerInputContextCompleted => this
        case StartContainerStarted => copy(status = StartingContainer)
        case StartContainerCompleted => this
        case LogRetrievalStarted => copy(status = RetrievingLogs)
        case LogRetrievalCompleted => this
        case LogEntryAdded(logEntryRef) => copy(lastLogChunk = logEntryRef.chunk, lastLogTimestamp = logEntryRef.lastEntryTimestamp)
        case DirectivesAddedEvent(_) => this // TODO should we store directives into the state?
        case WaitForContainerStarted => copy(status = RunningContainer)
        case WaitForContainerCompleted(statusCode) => copy(containerStatusCode = statusCode)
        case CreateContainerOutputContextStarted => copy(status = CreatingContainerOutputContext)
        case CreateContainerOutputContextCompleted(outputResult) => copy(outputResultOrExceptionTrace = outputResult)
        case RemoveContainerStarted => copy(status = Stopping)
        case RemoveContainerCompleted => this
        case TerminateJobStarted => copy(status = TerminatingJob)
        case TerminateJobCompleted => this
        case OperationFailed(msg) => copy(outputResultOrExceptionTrace = msg)
        case ConnectionFailed(msg) =>
          copy(status = ConnectingToDocker, outputResultOrExceptionTrace = msg, dockerOperationRecoveryEvent = status.recoverTo())
      }
      newState.validate()
    }

    private def validate(): DockerJobExecutorState = {
      if (jobData == null) {
        throw new DockerExecutorIllegalState("Job data is null and we cannot process such message")
      }
      this
    }
  }

  // Executor statuses
  sealed trait ExecutorStatus {
    def recoverTo(): Option[DockerJobOperationStarted]
  }

  case object Initial extends ExecutorStatus {
    override def recoverTo(): Option[DockerJobOperationStarted] = None
  }

  case object StartingJob extends ExecutorStatus {
    override def recoverTo(): Option[DockerJobOperationStarted] = None
  }

  case object ConnectingToDocker extends ExecutorStatus {
    override def recoverTo(): Option[DockerJobOperationStarted] = None
  }

  case object PullingImage extends ExecutorStatus {
    override def recoverTo(): Option[DockerJobOperationStarted] = Some(PullImageStarted)
  }

  case object CreatingContainer extends ExecutorStatus {
    override def recoverTo(): Option[DockerJobOperationStarted] = Some(CreateContainerStarted)
  }

  case object CreatingContainerInputContext extends ExecutorStatus {
    override def recoverTo(): Option[DockerJobOperationStarted] = Some(CreateContainerInputContextStarted)
  }

  case object StartingContainer extends ExecutorStatus {
    override def recoverTo(): Option[DockerJobOperationStarted] = Some(StartContainerStarted)
  }

  case object RetrievingLogs extends ExecutorStatus {
    override def recoverTo(): Option[DockerJobOperationStarted] = Some(LogRetrievalStarted)
  }

  case object RunningContainer extends ExecutorStatus {
    override def recoverTo(): Option[DockerJobOperationStarted] = Some(WaitForContainerStarted)
  }

  case object CreatingContainerOutputContext extends ExecutorStatus {
    override def recoverTo(): Option[DockerJobOperationStarted] = Some(CreateContainerOutputContextStarted)
  }

  case object TerminatingJob extends ExecutorStatus {
    override def recoverTo(): Option[DockerJobOperationStarted] = Some(TerminateJobStarted)
  }

  case object Stopping extends ExecutorStatus {
    override def recoverTo(): Option[DockerJobOperationStarted] = Some(RemoveContainerStarted)
  }

  // Events
  sealed trait DockerJobEvent extends SerializableMsg

  sealed trait RecoverableEvent extends DockerJobEvent

  sealed trait UnRecoverableEvent extends DockerJobEvent

  case class JobStarted(dockerOptions: DockerOptions, jobData: ContainerJobData) extends RecoverableEvent

  case class LogEntryAdded(logEntryRef: LogEntryRef) extends UnRecoverableEvent

  case class DirectivesAddedEvent(directives: Seq[Directive]) extends UnRecoverableEvent

  sealed trait DockerJobOperationStarted extends RecoverableEvent

  sealed trait DockerJobOperationCompleted extends RecoverableEvent

  sealed trait DockerJobOperationFailed extends RecoverableEvent

  case object DockerConnectRetryFailed extends DockerJobOperationFailed

  case class ConnectionFailed(errorMessage: String) extends DockerJobOperationFailed

  case class OperationFailed(errorMessage: String) extends DockerJobOperationFailed

  case object PullImageStarted extends DockerJobOperationStarted

  case object PullImageCompleted extends DockerJobOperationCompleted

  case object CreateContainerStarted extends DockerJobOperationStarted

  case class CreateContainerCompleted(containerId: String) extends DockerJobOperationCompleted

  case object CreateContainerInputContextStarted extends DockerJobOperationStarted

  case object CreateContainerInputContextCompleted extends DockerJobOperationCompleted

  case object StartContainerStarted extends DockerJobOperationStarted

  case object StartContainerCompleted extends DockerJobOperationCompleted

  case object LogRetrievalStarted extends DockerJobOperationStarted

  case object LogRetrievalCompleted extends DockerJobOperationCompleted

  case object WaitForContainerStarted extends DockerJobOperationStarted

  case class WaitForContainerCompleted(statusCode: Int) extends DockerJobOperationCompleted

  case object CreateContainerOutputContextStarted extends DockerJobOperationStarted

  case class CreateContainerOutputContextCompleted(outputResult: String) extends DockerJobOperationCompleted

  case object RemoveContainerStarted extends DockerJobOperationStarted

  case object RemoveContainerCompleted extends DockerJobOperationCompleted

  case object TerminateJobStarted extends DockerJobOperationStarted

  case object TerminateJobCompleted extends DockerJobOperationCompleted


  // Commands
  sealed trait DockerJobCommand extends SerializableMsg {
    def jobId: JobId
  }

  case class DockerPingResult(jobId: JobId, success: Boolean) extends DockerJobCommand

  case class StartJob(dockerOptions: DockerOptions, jobData: ContainerJobData) extends DockerJobCommand {
    override def jobId: JobId = jobData.jobId
  }

  case class ResumeJob(jobId: JobId) extends DockerJobCommand

  case class TerminateJob(jobId: JobId) extends DockerJobCommand

  case class StopExecutor(jobId: JobId) extends DockerJobCommand

  case class AddLogEntry(logEntry: LogEntry, storedEntryUri: URI) extends DockerJobCommand {
    override def jobId: JobId = logEntry.jobId
  }

  sealed trait DockerOperation extends DockerJobCommand

  case class PullImage(jobId: JobId) extends DockerOperation

  case class CreateContainer(jobId: JobId) extends DockerOperation

  case class CreateContainerInputContext(jobId: JobId) extends DockerOperation

  case class StartContainer(jobId: JobId) extends DockerOperation

  case class RetrieveLogs(jobId: JobId) extends DockerOperation

  case class WaitForContainer(jobId: JobId) extends DockerOperation

  case class CreateContainerOutputContext(jobId: JobId) extends DockerOperation

  case class RemoveContainer(jobId: JobId) extends DockerOperation

  sealed trait DockerOperationResult extends DockerJobCommand {
    def event: DockerJobEvent
  }

  case class DockerOperationSuccess(jobId: JobId, event: DockerJobOperationCompleted) extends DockerOperationResult

  case class DockerOperationFailure(jobId: JobId, event: DockerJobOperationFailed) extends DockerOperationResult

}
