package com.xebialabs.xlrelease.runner.docker.actors

import akka.actor.ActorRef
import akka.cluster.sharding.ShardRegion.EntityId
import akka.cluster.sharding.{ClusterSharding, ClusterShardingSettings, ShardRegion}
import com.typesafe.config.Config
import com.xebialabs.xlplatform.cluster.ClusterMode
import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.runner.actors.JobExecutorActorFactory
import com.xebialabs.xlrelease.runner.docker.actors.DockerJobExecutorActor._
import com.xebialabs.xlrelease.runner.docker.domain.{DockerJobRunner, DockerOptions, RegistryOptions}
import com.xebialabs.xlrelease.runner.domain.{ContainerJobData, JobId}
import com.xebialabs.xlrelease.support.akka.spring.SpringExtension
import org.springframework.context.annotation.Profile
import org.springframework.stereotype.Component

import javax.annotation.PostConstruct


@Profile(Array(ClusterMode.STANDALONE))
@Component
class NonClusteredDockerJobExecutorActorFactory(springExtension: SpringExtension) extends DockerJobExecutorActorFactory {

  private lazy val dockerJobProcessingActor: ActorRef = springExtension.actorOf(classOf[DockerJobExecutorProcessingActor], name = "docker-job-processing-actor")

  override def create: FactoryMethod = {
    case () => dockerJobProcessingActor
  }
}

@Profile(Array(ClusterMode.FULL))
@Component
class ClusteredDockerJobExecutorActorFactory(xlrConfig: XlrConfig, systemHolder: ActorSystemHolder, springExtension: SpringExtension)
  extends DockerJobExecutorActorFactory {
  private def actorSystem = systemHolder.actorSystem

  @PostConstruct
  def init() = {
    shardRegion
  }

  lazy val shardRegion = {
    val sharding = ClusterSharding(actorSystem)
    val originalConfig = actorSystem.settings.config.getConfig("akka.cluster.sharding")
    val shardingConfig: Config = xlrConfig.xl.getConfig("job-runner.akka.cluster.sharding").withFallback(originalConfig)
    val shardingSettings = ClusterShardingSettings(shardingConfig)

    val actorProps = springExtension.props(classOf[DockerJobExecutorProcessingActor])
    val shardRegion = sharding.start(
      typeName = DockerJobExecutorActor.SHARDING_TYPE_NAME,
      entityProps = actorProps,
      settings = shardingSettings,
      extractEntityId = extractEntityId,
      extractShardId = extractShardId
    )
    shardRegion
  }

  override def create: FactoryMethod = {
    case () => shardRegion
  }

  private def extractEntityId: ShardRegion.ExtractEntityId = {
    case msg: DockerJobCommand =>
      val entityId = actorName(msg.jobId)
      (entityId, msg)
  }

  private def extractShardId: ShardRegion.ExtractShardId = {
    case msg: DockerJobCommand =>
      val entityId = actorName(msg.jobId)
      shardId(entityId)
    case ShardRegion.StartEntity(entityId) =>
      shardId(entityId)
  }


  def shardId(entityId: EntityId): String = {
    val numberOfShards = xlrConfig.sharding.numberOfReleaseShards
    (math.abs(entityId.hashCode) % numberOfShards).toString
  }
}

sealed trait DockerJobExecutorActorFactory extends JobExecutorActorFactory {

  override def start: StartMethod = {
    case (jobRunner: DockerJobRunner, jobData: ContainerJobData) =>
      val registryOptions = RegistryOptions(jobData.registryUrl)
      val dockerOptions = DockerOptions(jobRunner.getId, jobRunner.getHost, jobRunner.capacity, registryOptions)
      val actorRef = create()
      actorRef ! StartJob(dockerOptions, jobData)
      actorRef
  }

  override def resume: ResumeMethod = {
    case jobId: JobId =>
      val actorRef = create()
      actorRef ! ResumeJob(jobId)
      actorRef
  }

  override def abort: AbortMethod = {
    case jobId: JobId =>
      val actorRef = create()
      actorRef ! TerminateJob(jobId)
      actorRef
  }
}
