#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#
from xlropenshift.client.openshift_client import OpenShiftClient
import kubernetes.client

class ImageTagHelper(object):
    def __init__(self, python_script, server):
        self.__python_script = python_script
        self.__client = OpenShiftClient(server)

        # OpenShift dynamic client
        self.__dyn_client = self.__client.dyn_client()

    def tag_image(self):

        # Accessing source image tag details
        source_image_tag = self.get_source_image_tag_details()

        # Returning a ResourceInstance object
        v1_image_stream_tag = self.__dyn_client.resources.get(api_version='image.openshift.io/v1', kind='ImageStreamTag')

        if self.get_target_image_tag_details():
            v1_image_stream_tag.patch(
                name=self.__get_target_image_name_and_tag(),
                namespace=self.__get_target_project_name(),
                body=self.__create_image_stream_tag_body(source_image_tag)
            )
        else:
            v1_image_stream_tag.create(
                namespace=self.__get_target_project_name(),
                body=self.__create_image_stream_tag_body(source_image_tag)
            )

        print "Image tag created from {}/{} to {}/{}".format(self.__get_source_project_name(),
                                                             self.__get_source_image_name_and_tag(),
                                                             self.__get_target_project_name(),
                                                             self.__get_target_image_name_and_tag())

    def __create_image_stream_tag_body(self, source_image_tag):
        image_ref = {
            "kind": "Image",
            "apiVersion": "image.openshift.io/v1",
            "dockerImageLayers": source_image_tag.image.dockerImageLayers,
            "dockerImageMetadata": {
                "Created": source_image_tag.image.dockerImageMetadata.Created,
                "Id": source_image_tag.image.dockerImageMetadata.Id,
                "apiVersion": source_image_tag.image.dockerImageMetadata.apiVersion,
                "kind": source_image_tag.image.dockerImageMetadata.kind
            },
            "dockerImageMetadataVersion": source_image_tag.image.dockerImageMetadataVersion,
            "dockerImageReference": source_image_tag.image.dockerImageReference,
            "metadata": kubernetes.client.V1ObjectMeta(name=source_image_tag.image.metadata.name, creation_timestamp=source_image_tag.image.metadata.creationTimestamp)
        }

        tag_from = kubernetes.client.V1ObjectReference(kind='ImageStreamImage',
                                                       namespace=self.__get_source_project_name(),
                                                       name='{}@{}'.format(self.__get_source_image_name(),
                                                                           source_image_tag.image.metadata.name))
        tag_ref = {
            "kind": "TagReference",
            "apiVersion": "image.openshift.io/v1",
            "name": self.__get_target_image_tag(),
            "from": tag_from
        }

        lookup_policy_ref = {
            "kind": "ImageLookupPolicy",
            "apiVersion": "image.openshift.io/v1",
            "local": source_image_tag.lookupPolicy.local
        }

        body = {
            "kind": "ImageStreamTag",
            "apiVersion": "image.openshift.io/v1",
            "image": image_ref,
            "metadata": kubernetes.client.V1ObjectMeta(name=self.__get_target_image_name_and_tag()),
            "tag": tag_ref,
            "generation": source_image_tag.generation,
            "lookupPolicy": lookup_policy_ref
        }
        return body

    def get_source_image_tag_details(self):
        # If tag object is null it will throw 'Exception during execution: ValueError: Invalid value for tag, must not be None'

        v1_image_stream_tag = self.__dyn_client.resources.get(api_version='image.openshift.io/v1', kind='ImageStreamTag')

        # Accessing all imagestreamtags from source project
        image_tag_response = v1_image_stream_tag.get(
             namespace=self.__get_source_project_name())

        for item in image_tag_response.items:
            if item.metadata.name == self.__get_source_image_name_and_tag():
                return item
        raise RuntimeError(
            "Source image tag not found for : {} on project {} ".format(self.__get_source_image_name_and_tag(),
                                                                        self.__get_source_project_name()))

    def get_target_image_tag_details(self):

        v1_image_stream_tag = self.__dyn_client.resources.get(api_version='image.openshift.io/v1', kind='ImageStreamTag')

        # Accessing all imagestreamtags from target project
        image_tag_response = v1_image_stream_tag.get(
            namespace=self.__python_script.getProperty('targetProjectName'))

        for item in image_tag_response.items:
            if item.metadata.name == self.__get_target_image_name_and_tag():
                return item

    def __get_source_project_name(self):
        return self.__python_script.getProperty('sourceProjectName')

    def __get_target_project_name(self):
        return self.__python_script.getProperty('targetProjectName') if self.__python_script.getProperty(
            'targetProjectName') else self.__get_source_project_name()

    def __get_source_image_name(self):
        return self.__python_script.getProperty('sourceImageNameAndTag').split(':')[0]

    def __get_source_image_tag(self):
        splits = self.__python_script.getProperty('sourceImageNameAndTag').split(':')
        return splits[1] if (len(splits) > 1) else 'latest'

    def __get_source_image_name_and_tag(self):
        return '{}:{}'.format(self.__get_source_image_name(), self.__get_source_image_tag())

    def __get_target_image_name(self):
        return self.__python_script.getProperty('targetImageNameAndTag').split(':')[0]

    def __get_target_image_tag(self):
        splits = self.__python_script.getProperty('targetImageNameAndTag').split(':')
        return splits[1] if (len(splits) > 1) else 'latest'

    def __get_target_image_name_and_tag(self):
        return '{}:{}'.format(self.__get_target_image_name(), self.__get_target_image_tag())

    @staticmethod
    def print_log(msg):
        print "```"
        print msg
        print "```"

    def logout(self):
        self.__client.logout()