'use strict';

(function () {

    function ReleaseHealthController($scope, ReleasesService, FlagService, TasksService, Planner) {
        var vm = this;
        vm.message = '';
        vm.light = 'green';

        vm.releaseFlag = {
            status: '',
            comment: ''
        };
        vm.flaggedTasks = 0;
        vm.failedTasks = 0;
        vm.totalTasks = 0;
        vm.completedTasks = 0;
        vm.delay = '';

        function detectFailedRelease(release) {
            if (ReleasesService.isReleaseFailing(release) || ReleasesService.isReleaseFailed(release)) {
                vm.light = 'red';
            }
        }

        function detectReleaseFlags(release) {
            if (FlagService.isFlaggedAtRisk(release)) {
                vm.light = 'red';
                vm.releaseFlag = release.releaseFlag;
            }

            if (FlagService.isFlaggedAttentionNeeded(release) && vm.light != 'red' && release.releaseFlag.status === 'ATTENTION_NEEDED') {
                vm.light = 'yellow';
                vm.releaseFlag = release.flag;
            }
        }

        function detectDelay(release) {
            var diff = calculateDelay(release);
            vm.delay = moment.duration(diff);
            if (diff > 0 && vm.light != 'red') {
                vm.light = 'yellow';
            }
        }

        function detectFlaggedTasks(release) {
            var allTasks = ReleasesService.getAllTasks(release);
            _.forEach(allTasks, function (task) {
                if (FlagService.isItemFlagged(task) && !TasksService.isTaskReadOnly(task)) {
                    if (FlagService.isFlaggedAtRisk(task)) {
                        vm.light = 'red';
                    } else if (FlagService.isFlaggedAttentionNeeded(task) && vm.light != 'red') {
                        vm.light = 'yellow';
                    }
                    vm.flaggedTasks++;
                }
            });
        }

        function detectFailedAndTotalTasks(release) {
            var leafTasks = ReleasesService.getLeafTasks(release);
            vm.totalTasks = leafTasks.length;
            vm.failedTasks = _.filter(leafTasks, TasksService.isTaskFailed).length;
            if (vm.failedTasks > 0) {
                vm.light = 'red';
            }
            vm.completedTasks = _.filter(leafTasks, TasksService.isTaskReadOnly).length;
        }

        function detectMessage() {
            switch (vm.light) {
                case 'red':
                    vm.message = 'Release at risk';
                    break;
                case 'yellow':
                    vm.message = 'Attention needed';
                    break;
                case 'green':
                    vm.message = 'OK';
                    break;
            }
        }

        function calculateCurrentState(release) {
            detectFailedRelease(release);
            detectReleaseFlags(release);
            detectDelay(release);
            detectFlaggedTasks(release);
            detectFailedAndTotalTasks(release);
            detectMessage();
        }

        function calculateDelay(release) {
            Planner.setRelease(release);
            Planner.setReleaseTree(release.tree);
            var planContainer = Planner.makePlan(false);
            var releasePlan = planContainer.plan;
            var expectedEndDate = planContainer.plannedEndDate;
            var releasePlannedEndDate = releasePlan.displayEndDate;
            return expectedEndDate.diff(releasePlannedEndDate);
        }

        calculateCurrentState($scope.release);
    }

    ReleaseHealthController.$inject = ['$scope', 'ReleasesService', 'FlagService', 'TasksService', 'Planner'];

    angular.module('xlrelease').controller('summary.ReleaseHealthController', ReleaseHealthController);
})();