/*
 * Decompiled with CFR 0.152.
 */
package com.xebialabs.xlrelease.plugins.remotecompletion.email;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlrelease.plugins.remotecompletion.ci.CiUtils;
import com.xebialabs.xlrelease.plugins.remotecompletion.ci.ImapServer;
import com.xebialabs.xlrelease.plugins.remotecompletion.email.ImapTokenHelper;
import com.xebialabs.xlrelease.plugins.remotecompletion.util.FunctionalUtils;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;
import jakarta.mail.Folder;
import jakarta.mail.Message;
import jakarta.mail.MessagingException;
import jakarta.mail.Session;
import jakarta.mail.Store;
import jakarta.mail.internet.MimeMessage;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.stream.Collectors;
import org.eclipse.angus.mail.imap.IMAPStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EmailStore {
    private static final Logger logger = LoggerFactory.getLogger(EmailStore.class);
    private static final int MAIL_TIMEOUT = 30000;
    private static final String MAIL_IMAP_CONNECTION_TIMEOUT = "mail.imap.connectiontimeout";
    private static final String MAIL_IMAP_TIMEOUT = "mail.imap.timeout";
    private static final String GRAPH_API_BASE = "https://graph.microsoft.com/v1.0";
    private HttpClient httpClient;
    private String accessToken;
    private Optional<Store> store = Optional.empty();
    protected ImapServer imapServer;
    private Optional<Folder> inbox = Optional.empty();
    private ImapTokenHelper imapTokenHelper;

    public EmailStore(ImapServer imapServer, ImapTokenHelper imapTokenHelper) throws Exception {
        this.imapServer = imapServer;
        this.imapTokenHelper = imapTokenHelper;
        if (FunctionalUtils.useGraph(imapServer)) {
            logger.debug("Using Microsoft Graph API for email access");
            this.initializeUsingGraph(false);
        } else {
            logger.debug("Using IMAP for email access");
            this.initializeStore(imapServer, "imap");
            this.initializeInbox(false);
        }
    }

    private void initializeUsingGraph(boolean decryptPassword) throws Exception {
        this.imapTokenHelper.refreshTokenIfRequired(this.imapServer, decryptPassword);
        this.accessToken = PasswordEncrypter.getInstance().decrypt(this.imapServer.getAccessToken());
        this.httpClient = HttpClient.newBuilder().connectTimeout(Duration.ofSeconds(20L)).build();
    }

    public List<String> getInboxMessageIdsUsingGraph() throws Exception {
        String url = String.format("%s/users/%s/mailFolders/inbox/messages?$select=id&$top=100", GRAPH_API_BASE, this.imapServer.getFromAddress());
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(url)).header("Authorization", "Bearer " + this.accessToken).header("Accept", "application/json").GET().build();
        HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
        if (response.statusCode() != 200) {
            throw new RuntimeException("Failed to get message list: HTTP " + response.statusCode() + " - " + response.body());
        }
        ArrayList<String> ids = new ArrayList<String>();
        ObjectMapper mapper = new ObjectMapper();
        JsonNode root = mapper.readTree(response.body());
        for (JsonNode message : root.get("value")) {
            ids.add(message.get("id").asText());
        }
        return ids;
    }

    private byte[] getMessageMimeValueUsingGraph(String messageId) throws Exception {
        String url = String.format("%s/users/%s/messages/%s/$value", GRAPH_API_BASE, this.imapServer.getFromAddress(), messageId);
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(url)).header("Authorization", "Bearer " + this.accessToken).GET().build();
        HttpResponse<byte[]> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofByteArray());
        if (response.statusCode() != 200) {
            throw new RuntimeException("Failed to get MIME content for message " + messageId + ": HTTP " + response.statusCode());
        }
        return response.body();
    }

    public Message[] getInboxMessagesUsingGraph() throws Exception {
        List<String> messageIds = this.getInboxMessageIdsUsingGraph();
        Properties props = new Properties();
        Session session = Session.getInstance((Properties)props, null);
        ArrayList<MimeMessage> messages = new ArrayList<MimeMessage>();
        logger.debug("Found {} messages. Fetching MIME content...", (Object)messageIds.size());
        for (String id : messageIds) {
            try {
                logger.debug("Fetching MIME content for message ID: {}", (Object)id);
                byte[] mimeContent = this.getMessageMimeValueUsingGraph(id);
                ByteArrayInputStream is = new ByteArrayInputStream(mimeContent);
                MimeMessage mimeMessage = new MimeMessage(session, (InputStream)is);
                mimeMessage.setHeader("X-Graph-Id", id);
                messages.add(mimeMessage);
            }
            catch (MessagingException me) {
                throw new RuntimeException("Failed to parse message with ID " + id, me);
            }
            catch (Exception e) {
                throw new RuntimeException("Failed to fetch message with ID " + id, e);
            }
        }
        logger.debug("Successfully fetched and parsed {} messages.", (Object)messages.size());
        return messages.toArray(new Message[0]);
    }

    public void deleteMessageUsingGraph(String messageId) throws Exception {
        String url = String.format("%s/users/%s/messages/%s", GRAPH_API_BASE, this.imapServer.getFromAddress(), messageId);
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(url)).header("Authorization", "Bearer " + this.accessToken).DELETE().build();
        HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
        if (response.statusCode() != 204) {
            throw new RuntimeException("Failed to delete message " + messageId + ": HTTP " + response.statusCode() + " - " + response.body());
        }
        logger.debug("Successfully deleted message: {}", (Object)messageId);
    }

    public ImapServer getImapServer() {
        return this.imapServer;
    }

    public void close() {
        logger.debug("Closing the IMAPStore");
        this.store.ifPresent(store -> {
            try {
                store.close();
            }
            catch (Exception e) {
                logger.warn("Encountered an exception while closing IMAPStore", (Throwable)e);
            }
        });
    }

    public EmailStore(ConfigurationRepository configurationRepository, ImapTokenHelper imapTokenHelper) throws Exception {
        this.imapTokenHelper = imapTokenHelper;
        Optional<Object> localImapServer = Optional.empty();
        try {
            localImapServer = CiUtils.getImapServer(configurationRepository);
        }
        catch (Exception exception) {
            // empty catch block
        }
        this.imapServer = localImapServer.orElse((Object)new ImapServer());
        if (!this.imapServer.isValid()) {
            logger.debug("Could not find a valid mail server configuration. Skipping mail synchronisation.");
            return;
        }
        logger.debug("Mail server configuration found. Started mail server synchronisation. {}", (Object)this.imapServer);
        if (FunctionalUtils.useGraph(this.imapServer)) {
            logger.debug("Using Microsoft Graph API for email access");
            this.initializeUsingGraph(true);
        } else {
            logger.debug("Using IMAP for email access");
            this.initializeStore(this.imapServer, "imap");
            this.initializeInbox(true);
        }
    }

    public boolean isWhitelistEnabled() {
        return this.imapServer.isWhitelistEnabled();
    }

    public List<String> getWhitelist() {
        return this.imapServer.getWhitelist();
    }

    public boolean hasSortCapability() {
        try {
            return ((IMAPStore)this.store.get()).hasCapability("SORT*");
        }
        catch (MessagingException e) {
            logger.error("Encountered an exception while checking sort capability : " + FunctionalUtils.getNestedExceptionsMessagesFunction.apply(e));
            return false;
        }
    }

    public String getFromAddress() {
        return this.imapServer.getFromAddress();
    }

    private void initializeStore(ImapServer server, String protocol) throws Exception {
        Properties props = this.getImapProperties(server);
        Session session = Session.getInstance((Properties)props);
        this.store = Optional.ofNullable(session.getStore(protocol));
    }

    private void initializeInbox(boolean decryptPassword) throws Exception {
        if (this.imapServer.isUseOAuth2()) {
            this.imapTokenHelper.refreshTokenIfRequired(this.imapServer, decryptPassword);
            String accessToken = PasswordEncrypter.getInstance().decrypt(this.imapServer.getAccessToken());
            this.store.get().connect(this.imapServer.getHost(), this.imapServer.getPort(), this.imapServer.getFromAddress(), accessToken);
        } else {
            String password = decryptPassword ? PasswordEncrypter.getInstance().decrypt(this.imapServer.getPassword()) : this.imapServer.getPassword();
            this.store.get().connect(this.imapServer.getHost(), this.imapServer.getPort(), this.imapServer.getUsername(), password);
        }
        Folder inbox = this.store.get().getFolder("INBOX");
        inbox.open(2);
        this.inbox = Optional.of(inbox);
    }

    public Optional<Folder> getInbox() throws Exception {
        if (!this.inbox.isPresent() && this.store.isPresent()) {
            this.initializeInbox(true);
        }
        this.inbox.ifPresent(folder -> {
            try {
                if (!folder.isOpen()) {
                    folder.open(2);
                }
            }
            catch (MessagingException e) {
                logger.error("Unable to open inbox for read/write mode", (Throwable)e);
            }
        });
        return this.inbox;
    }

    private Properties getImapProperties(ImapServer imapServer) {
        Properties props = System.getProperties();
        String mailTimeout = String.valueOf(30000);
        if (!props.containsKey(MAIL_IMAP_CONNECTION_TIMEOUT)) {
            props.setProperty(MAIL_IMAP_CONNECTION_TIMEOUT, mailTimeout);
        }
        if (!props.containsKey(MAIL_IMAP_TIMEOUT)) {
            props.setProperty(MAIL_IMAP_TIMEOUT, mailTimeout);
        }
        if (imapServer.isTls()) {
            props.setProperty("mail.imap.socketFactory.class", "javax.net.ssl.SSLSocketFactory");
            props.setProperty("mail.imap.socketFactory.fallback", "false");
        } else {
            props.setProperty("mail.imap.socketFactory.class", "javax.net.SocketFactory");
        }
        if (imapServer.isUseOAuth2()) {
            props.put("mail.imap.auth.mechanisms", "XOAUTH2");
        } else {
            props.remove("mail.imap.auth.mechanisms");
        }
        props.setProperty("mail.imap.ssl.checkserveridentity", "false");
        logger.trace("Connection properties passed to javamail: {}", props.entrySet().stream().filter(e -> e.getKey().toString().startsWith("mail")).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)));
        return props;
    }
}

