'use strict';
(function () {

    var TaskProgressTileViewController = function ($scope, XlrTileHelper, TasksService, ReleaseGridService) {
        var vm = this;
        var tile;
        if ($scope.xlrTile) {
            // summary mode
            vm.release = $scope.release;
            tile = $scope.xlrTile.tile;
        } else {
            // details mode
            tile = $scope.xlrTileDetailsCtrl.tile;
            vm.release = $scope.xlrTileDetailsCtrl.release;
        }
        vm.themes = [
            {value : "green", text : "Green (default)"},
            {value : "yellow", text : "Yellow"},
            {value : "blue", text : "Blue"},
            {value : "purple", text : "Purple"},
            {value : "darkGreen", text : "Dark green"}
        ];

        vm.chartOptions = {
            topTitleText: function(data) {
                return data.completed + "/" + (data.completed + data.planned);
            },
            bottomTitleText: "tasks",
            legend: {
                borderRadius: 0
            },
            series: function (data) {
                return [{
                    name: 'Tasks',
                    data: [{
                        y: data.completed,
                        name: "Completed",
                        color: getThemeColor(tile.configurationProperties.theme)['Completed']
                    }, {
                        y: data.planned,
                        name: "Planned",
                        color: getThemeColor(tile.configurationProperties.theme)['Planned']
                    }]
                }];
            },
            showLegend: true,
            donutThickness: '90%'
        };

        refresh();

        function getThemeColor(theme) {
            switch (theme) {
                case "yellow":
                    return {"Completed":"#ffd025","Planned":"#d6cead"};
                case "blue":
                    return {"Completed":"#009cd8","Planned":"#bad3dc"};
                case "purple":
                    return {"Completed":"#8d5495","Planned":"#d9c5dc"};
                case "darkGreen":
                    return {"Completed":"#358f6d","Planned":"#bad9cd"};
                default:
                    return {"Completed":"#5DAE3F","Planned":"#DDD"};
            }
        }

        function refresh() {
            vm.tags = tile.configurationProperties.tags;
            vm.taskWithTags = getTaskWithTags(vm.tags);

            var completedTasks = calculateCompletedTasks(vm.taskWithTags);
            vm.chartData = {
                completed: completedTasks.length,
                planned: vm.taskWithTags.length - completedTasks.length
            };
            vm.progress = calculateProgress(completedTasks.length, vm.taskWithTags.length);
            vm.gridOptions = createGridOptions(vm.taskWithTags);
        }

        function isTask(entity) {
            return !('xlrelease.Release' === entity.type || 'xlrelease.Phase' === entity.type);
        }

        function calculateCompletedTasks(tasks) {
            return _.filter(tasks, TasksService.isTaskReadOnly);
        }

        function calculateProgress(completed, total) {
            var progress = 0;
            if (angular.isNumber(completed) && angular.isNumber(total) && total != 0) {
                progress = completed/total * 100;
            }
            return progress;
        }

        function getTaskWithTags(tags) {
            return ReleaseGridService.getGridItems(vm.release).filter(isTask).filter (function (task) {
                var taskWithTag = false;
                tags.value.forEach(function (tag) {
                    task.planItem.tags.forEach(function (taskTag) {
                        if (tag.toLowerCase() === taskTag.toLowerCase()) {taskWithTag = true;}
                    })
                });
                return taskWithTag;
            });
        }

        function getAllTagsInRelease () {
            var taskTags = ReleaseGridService.getGridItems(vm.release).filter(isTask).filter (function (task) {
                return task.planItem.tags.length > 0;
            }).map(function (task) { return task.planItem.tags; });
            var tags  = [];
            taskTags.forEach(function (tagList) {
                tagList.forEach(function (tag) {
                    var t = tag.toLowerCase();
                    if (tags.indexOf(t) == -1)
                        tags.push(t);
                });
            });
            return tags;
        }
        vm.getAllTagsInRelease = getAllTagsInRelease;

        function getUnconfiguredTagsInRelease(existingTags) {
            var tags =  vm.getAllTagsInRelease().filter(function (tag) {
                return existingTags.indexOf(tag) == -1;
            });
            return tags.sort();
        }
        vm.getUnconfiguredTagsInRelease = getUnconfiguredTagsInRelease;

        function isTagAvailableInRelease (tagName) {
            var allTaskTags = vm.getAllTagsInRelease();
            return allTaskTags.indexOf(tagName.toLowerCase()) != -1;
        }
        vm.isTagAvailableInRelease = isTagAvailableInRelease;

        function createGridOptions(tasks) {
            var dateFilter = 'mediumDateShortTime';
            var columnDefs = [
                {
                    displayName: "Name",
                    field: "title",
                    filterHeaderTemplate: "<div data-ng-include=\"'partials/releases/grid/templates/name-filter-template.html'\"></div>",
                    enableColumnMenu: false,
                    width: '30%'
                },
                {
                    displayName: "Status",
                    field: "status",
                    cellTemplate: "static/@project.version@/include/TaskProgressTile/grid/status-cell-template.html",
                    enableColumnMenu: false,
                    width: '10%'
                },
                {
                    displayName: "Start Date",
                    field: "startDate",
                    cellFilter: dateFilter,
                    filterHeaderTemplate: "<div ng-include=\"'partials/releases/grid/templates/date-filter-template.html'\"></div>",
                    enableFiltering: true,
                    enableColumnMenu: false,
                    width: '10%',
                    filter: {
                        placeholder: 'From:',
                        condition: function (searchTerm, cellValue) {
                            moment(cellValue).isAfter(moment(searchTerm).startOf('day'));
                        }
                    }
                },
                {
                    displayName: "End Date",
                    field: "endDate",
                    cellFilter: dateFilter,
                    filterHeaderTemplate: "<div ng-include=\"'partials/releases/grid/templates/date-filter-template.html'\"></div>",
                    enableColumnMenu: false,
                    enableFiltering: true,
                    filter: {
                        placeholder: 'To:',
                        condition: function (searchTerm, cellValue) {
                            moment(cellValue).isBefore(moment(searchTerm).endOf('day'))
                        }
                    },
                    width: '10%'
                },
                {
                    displayName: "Duration",
                    field: "duration",
                    enableColumnMenu: false,
                    filterCellFiltered: true,
                    cellFilter: 'prettyDuration',
                    width: '10%'
                },
                {
                    displayName: "Assigned To",
                    field: "assignedTo",
                    cellTemplate: "static/@project.version@/include/TaskProgressTile/grid/assigned-to-cell-template.html",
                    enableColumnMenu: false,
                    width: '12%'
                },
                {
                    displayName: "Tags",
                    field: "tags",
                    cellTemplate: "static/@project.version@/include/TaskProgressTile/grid/tags-cell-template.html",
                    enableColumnMenu: false,
                    width: '17%'
                }
            ];

            return XlrTileHelper.getGridOptions(tasks, columnDefs);
        }

        vm.refresh = refresh;
    };
    var injectParams = ['$scope','XlrTileHelper', 'TasksService', 'ReleaseGridService'];
    TaskProgressTileViewController.$inject = injectParams;
    angular.module('xlrelease.taskprogress.tile', []);
    angular.module('xlrelease.taskprogress.tile').controller('xlrelease.TaskProgressTileViewController', TaskProgressTileViewController);
    angular.module('xlrelease.taskprogress.tile').directive('taskProgressTags', function (xlrSetOfStringsWithVariablesDirective) {
        return angular.extend({}, xlrSetOfStringsWithVariablesDirective[0], { templateUrl:  'static/@project.version@/include/TaskProgressTile/directives/task-progress-tags-directive.html' });
    });

    angular.module('xlrelease.taskprogress.tile').directive('taskProgressDipSet', function (xlWidgetUtils, $timeout) {

        function deduplicate(list, field) {
            if (!list) return list;
            return _.uniqBy(list, function (item) {
                return field ? item[field] : item.title;
            });
        }

        function onModify($scope) {
            $timeout(function () {
                if (!$scope.readOnly && $scope.onChange) {
                    $scope.onChange();
                }
            });
        }

        return {
            restrict: 'A',
            scope: {
                ngModel: '=',
                metadata: '=',
                displayMode: '=?',
                readOnly: '=',
                onChange: '&'
            },
            templateUrl: function templateUrl(element, attrs) {
                return attrs.templateUrl || 'static/@project.version@/include/TaskProgressTile/directives/task-progress-dip-set-directive.html';
            },
            require: ['?^form', 'ngModel'],
            link: function link(scope, elem, attrs, ctrls) {
                var parentForm = ctrls[0];
                var ngModelCtrl = ctrls[1];
                scope.isRequired = scope.metadata.required;

                if (!scope.ngModel || !angular.isArray(scope.ngModel)) {
                    scope.ngModel = [];
                }

                scope.field = scope.metadata ? scope.metadata.field : undefined;

                if (!scope.field && !scope.readOnly) {
                    scope.$watch('$set', function (value) {
                        if (!value) return;
                        // we do not want to loose scope.ngModel reference
                        var v = _.map(scope.$set, 'title');
                        angular.copy(v, scope.ngModel);
                        ngModelCtrl.$setValidity('required', !xlWidgetUtils.hasRequiredError(scope.ngModel, scope.isRequired));
                    }, true);
                }

                scope.remove = function (item) {
                    var index = _.indexOf(scope.$set, item);
                    if (index > -1) {
                        scope.$set.splice(index, 1);
                        if (parentForm) {
                            parentForm.$setDirty();
                        }
                        onModify(scope);
                        scope.updateReleaseTags();
                    }
                };
            },
            controller: function controller($scope) {
                $scope.newItem = '';
                $scope.$watch('ngModel', function (newValue) {
                    if (!$scope.field) {
                        $scope.$set = deduplicate(_.map($scope.ngModel, function (item) {
                            return {title: item};
                        }));
                    } else {
                        $scope.$set = deduplicate($scope.ngModel, $scope.field);
                    }
                    $scope.updateReleaseTags();
                });

                $scope.$watch('readOnly', function () {
                    var readOnly = $scope.readOnly;
                    $scope.displayMode = readOnly ? 'visual' : 'creation';
                }, true);

                $scope.$watch('metadata', function () {
                    if ($scope.metadata) {
                        var defaultValue = $scope.metadata['default'];
                        if (!_.isNil(defaultValue) && !_.isNil(defaultValue)) {
                            defaultValue = defaultValue.substring(1, defaultValue.length - 1);
                            _.forEach(defaultValue.split(','), function (value) {
                                var item = {};
                                item[$scope.field ? $scope.field : 'title'] = value;
                                $scope.$set.push(item);
                            });
                            $scope.$set = deduplicate($scope.$set);
                        }
                    }
                });

                $scope.updateReleaseTags = function () {
                    $scope.releaseTags = $scope.$parent.$parent.vm.getUnconfiguredTagsInRelease(getExistingTags());
                };

                $scope.addItem = function ($event) {
                    if ($event && $event.isDefaultPrevented()) return;
                    //For Firefox and IE8+
                    //!$event For add button
                    if (!$event || $event.keyCode === 13 || $event.keyCode === 0 || $event.which === 13 || $event.which === 0) {
                        if ($event) {
                            $event.preventDefault();
                        }
                        $scope.error = '';
                        if ($scope.newItem === '') return;

                        if ($scope.$parent.$parent.vm.isTagAvailableInRelease($scope.newItem)) {
                            var item = {};
                            item[$scope.field ? $scope.field : 'title'] = $scope.newItem;
                            $scope.$set.push(item);
                            $scope.updateReleaseTags();
                        }
                        $scope.newItem = '';
                        onModify($scope);
                    }
                };

                function getExistingTags() {
                    return _.map($scope.$set, function (item) {
                        return $scope.field ? item[$scope.field] : item.title;
                    });
                }

                this.getScope = function () {
                    return $scope;
                };
            }
        };
    });
})();

